dojo.provide('xg.shared.util');

dojo.require('dojo.animation.Animation');

xg.shared.util = {
    safeBindUrl: function(url) {
        // IE and Safari don't like [ or ] in the request
        return url.replace(/\[/g, "%5B").replace(/\]/g, "%5D");
    },

    parseUrlParameters: function(url) {
        var urlParts   = url.split('?');
        var urlContent = new Object;

        if (urlParts.length > 1) {
            var urlPairs   = urlParts[1].split('&');

            for (var idx = 0; idx < urlPairs.length; idx++) {
                var kv = urlPairs[idx].split('=');

                urlContent[kv[0]] = kv[1];
            }
        }
        return urlContent;
    },

    /**
     * Parses Dojo widgets in the given DOM subtree.
     * Be careful not to parse widgets more than once.
     *
     * @param HTMLElement  The root of the subtree to parse, or null to parse the entire document.
     */
    parseWidgets: function(root) {
        // There may be widgets (PromotionLink) in the returned HTML
        // Even though http://dojotoolkit.org/pipermail/dojo-interest/2006-June/009622.html
        // recommends to just do dojo.widget.createWidget(this.body), that causes an error
        // since createWidget() calls .toLowerCase() on its first argument, and DOM nodes
        // don't have that method. So instead, we use some code from Brian at
        // http://kasparov.skife.org/blog/src/javascript/dojo/ which does what ContentPane does
        // [ David Sklar - 2006-12-20 ]
        var root = root || document.getElementsByTagName("body")[0] || document.body;
        var parser = new dojo.xml.Parse();
        var frag = parser.parseElement(root, null, true);
        // Use createComponents rather than createSubComponents, which doesn't parse the top-level node [Jon Aquino 2007-01-31]
        dojo.widget.getParser().createComponents(frag);
    },

    /**
     * Fixes two problems with img tags in IE:
     * <ul>
     *     <li>png transparency doesn't work</li>
     *     <li>img tags created with Javascript sometimes fail to display if they aren't preloaded</li>
     * </ul>
     *
     * @param imgs array of image tags, typically found using node.getElementsByTagName('img')
     * @param sync whether to fix the image synchronously or asynchronously i.e. whether it is more important to display the image
     *         as soon as possible (as in the case of the app header image) or to allow processing to continue (important when preloading
     *         all of the images of a panel -- you want the panel to appear ASAP, and the images can be filled in afterwards).
     *         Setting sync to true reduces the duration of the flash of opaque in the app header icon for some reason (from 1 second to instantaneous).
     * @param width optional. Used if img.width is 0, which sometimes happens
     * @param height optional. Used if img.height is 0, which sometimes happens
     * @see Guyon Roche, "JavaScript Image Preloader", http://www.webreference.com/programming/javascript/gr/column3/
     */
    fixImagesInIE: function(imgs, sync, width, height) {
        if (! (dojo.render.html.ie50 || dojo.render.html.ie55 || dojo.render.html.ie60)) { return; }
        dojo.lang.forEach(imgs, function(img) {
            if (dojo.lang.inArray(xg.shared.util.fixedImageURLs, img.src)) { return; }
            var fixImage = function() {
                var image = new Image();
                image.onload = image.onerror = image.onabort = function() {
                    img.src = img.src;
                    xg.shared.util.fixTransparencyInIEProper(img, width, height);
                    xg.shared.util.fixedImageURLs.push(img.src);
                }
                image.src = img.src;
            }
            if (sync) { fixImage(); }
            else { dojo.lang.setTimeout(fixImage, 0); }
        });
    },

    fixedImageURLs: [],

    /**
     * Consider fixImagesInIE instead of using this function directly; fixImagesInIE takes care of both
     * preloading images loaded with javascript and fixing transparency.
     *
     * img.width and img.height are sometimes 0 for some reason. If this happens, you can specify the
     * width and height explicitly.
     * @see fixImagesInIE
     */
    fixTransparencyInIEProper: function(img, width, height) {
        if (img && (dojo.render.html.ie50 || dojo.render.html.ie55 || dojo.render.html.ie60)
                && img.src.match(/png/) && dojo.style.isShowing(img)) {
            // The header image is sometimes distorted in IE (half-loaded, or incorrectly sized).
            // Preloading (as is done with fixImagesInIE) probably fixes this, but it also delays the image
            // from appearing for a couple of seconds. [Jon Aquino 2006-05-31]
            width = width ? width : img.width;
            height = height ? height : img.height;
            img.style.width = width + "px";
            img.style.height = height + "px";
            img.style.filter = "progid:DXImageTransform.Microsoft.AlphaImageLoader(src='" + img.src + "', sizingMethod='scale')";
            img.src = "/xn_resources/widgets/index/gfx/x.gif";
        }
        if (img) { img.style.visibility = 'visible'; }
    },

    fixTransparencyInIE: function(node) {
        if (dojo.render.html.ie50 || dojo.render.html.ie55 || dojo.render.html.ie60) {
            dojo.lang.forEach(node.getElementsByTagName('img'), function(img) {
                xg.shared.util.fixTransparencyInIEProper(img);
            });
        }
    },

    /**
     * Replaces newlines with <br />s, except on lines with certain HTML elements like <p>
     *
     * @param string s The original text or HTML
     * @return string  The text with <br />s inserted
     * @see xg_nl2br
     */
    nl2br: function(s) {
        s = s.replace(/\r\n/, "\n"); // IE [Jon Aquino 2007-03-31]
        result = '';
        dojo.lang.forEach(s.split("\n"), function (line) {
            result += line;
            // See Web Design Group, "HTML 4 Block-Level Elements", http://htmlhelp.com/reference/html40/block.html  [Jon Aquino 2007-03-31]
            // Keep this list in sync with xg_nl2br [Jon Aquino 2007-04-02]
            if (! line.match(/<.?OBJECT\b|<.?EMBED\b|<.?PARAM\b|<.?APPLET\b|<.?IFRAME\b|<.?SCRIPT\b|<.?BR\b|<.?ADDRESS\b|<.?BLOCKQUOTE\b|<.?CENTER\b|<.?DIR\b|<.?DIV\b|<.?DL\b|<.?FIELDSET\b|<.?FORM\b|<.?H1\b|<.?H2\b|<.?H3\b|<.?H4\b|<.?H5\b|<.?H6\b|<.?HR\b|<.?ISINDEX\b|<.?MENU\b|<.?NOFRAMES\b|<.?NOSCRIPT\b|<.?OL\b|<.?P\b|<.?PRE\b|<.?TABLE\b|<.?UL\b|<.?DD\b|<.?DT\b|<.?FRAMESET\b|<.?LI\b|<.?TBODY\b|<.?TD\b|<.?TFOOT\b|<.?TH\b|<.?THEAD\b|<.?TR\b/i)) {
                result += '<br />';
            }
            result += "\n";
        });
        return dojo.string.trim(result).replace(/(<br \/>)+$/, '');
    },

    /**
     * Shows the lightbox background.
     */
    showOverlay: function() {
        if (dojo.byId('xg_overlay')) { return; }
        var overlay = dojo.html.createNodesFromText('<div id="xg_overlay" style="height: 2000px;"></div>')[0];
        document.body.appendChild(overlay);
        overlay.style.height = this.getPageHeight() + 'px';
    },

    /**
     * Hides the lightbox background.
     */
    hideOverlay: function() {
        if (! dojo.byId('xg_overlay')) { return; }
        dojo.dom.removeNode(dojo.byId('xg_overlay'));
    },

    /**
     * Returns the distance between the top and bottom of the document's content, or the
     * viewport height, whichever is larger.
     */
    getPageHeight: function() {
        var yScroll;

        if (window.innerHeight && window.scrollMaxY) {
            yScroll = window.innerHeight + window.scrollMaxY;
        } else if (document.body.scrollHeight > document.body.offsetHeight){ // all but Explorer Mac
            yScroll = document.body.scrollHeight;
        } else { // Explorer Mac...would also work in Explorer 6 Strict, Mozilla and Safari
            yScroll = document.body.offsetHeight;
        }

        var windowHeight;
        if (self.innerHeight) {	// all except Explorer
            windowHeight = self.innerHeight;
        } else if (document.documentElement && document.documentElement.clientHeight) { // Explorer 6 Strict Mode
            windowHeight = document.documentElement.clientHeight;
        } else if (document.body) { // other Explorers
            windowHeight = document.body.clientHeight;
        }

        // for small pages with total height less then height of the viewport
        if(yScroll < windowHeight){
            pageHeight = windowHeight;
        } else {
            pageHeight = yScroll;
        }

        return pageHeight;
    },

    /**
     * Prevents the user from entering text longer than the given maxLength.
     * Note that this does not prevent the user from pasting in text that exceeds maxLength,
     * so you should validate that as well (on the client and the server).
     *
     * @param textarea  the textarea node
     * @param maxLength  the maximum allowed length
     * @see twitter.com
     */
    setMaxLength: function(textarea, maxLength) {
        dojo.event.connect(textarea, 'onkeypress', function(event) {
            if (event.which != dojo.event.browser.keys.KEY_BACKSPACE && textarea.value.length >= maxLength) {
                dojo.event.browser.stopEvent(event);
            }
        });
    },

    /**
	 * Notify user that he/she entered too much. You should check the actual length when user clicks submit.
     *
     * @param textarea  the textarea node
     * @param maxLength  the maximum length
     * @see twitter.com
     */
	setAdvisableMaxLength: function(textarea, maxLength) {
		var msgEl, timer, inErrorMode = 0, parent = textarea.parentNode, updateStatus = function() {
			if (!msgEl) {
				msgEl = document.createElement('small');
				msgEl.className = 'block';
				textarea.nextSibling ? parent.insertBefore(msgEl, textarea.nextSibling) : parent.appendChild(msgEl, textarea);
			}
			if (textarea.value.length > maxLength) {
				msgEl.innerHTML = xg.shared.nls.text('messageIsTooLong', textarea.value.length, maxLength);
				if (!inErrorMode) {
					dojo.html.addClass(textarea.parentNode, 'error');
				}
				inErrorMode = 1;
			} else {
				if (inErrorMode) {
					msgEl.innerHTML = ''; // can display the actual number of characters
					dojo.html.removeClass(textarea.parentNode, 'error');
				}
				inErrorMode = 0;
			}
			timer = 0;
		};
		var triggerUpdate = function () { if (!timer) timer = window.setTimeout(updateStatus, 50); }
        dojo.event.connect(textarea, 'onkeyup', triggerUpdate);
        dojo.event.connect(textarea, 'onkeypress', triggerUpdate);
		dojo.event.connect(textarea, 'onblur', triggerUpdate);
		dojo.event.connect(textarea, 'oncut', triggerUpdate);
		dojo.event.connect(textarea, 'onpaste', triggerUpdate);
		dojo.event.connect(textarea, 'onchange', triggerUpdate);
    },

    /**
     * Displays an alert message
     *
     * @param string alert message, or object with the following properties:
     *   bodyHtml  HTML for the dialog message (to be placed in a <p>)
     *   title     (optional) plain-text title, defaults to none
     *   param     (optional) okButtonText  text for the OK button; defaults to "OK"
     */
    alert: function(stringOrObject) {
        //  If there's already an alert up, remove it
        if (dojo.byId('xg_lightbox_alert')) {
            dojo.dom.removeNode(dojo.byId('xg_lightbox_alert'));
        }
        //  Is the argument a string or an object?
        if ((typeof stringOrObject) == 'string') {
            args = {bodyHtml: stringOrObject};
        } else {
            args = stringOrObject;
        }

        if (!args.okButtonText) {
            args.okButtonText = xg.shared.nls.text('ok');
        }
        var dialogHtml = dojo.string.trim(' \
                <div class="xg_floating_module" id="xg_lightbox_alert"> \
                    <div class="xg_floating_container"> \
                        <div class="xg_module_head ' + (args.title ? '' : 'notitle') + '"> \
                            ' + (args.title ? '<h2>' + dojo.string.escape('html', args.title) + '</h2>' : '') + ' \
                        </div> \
                        <div class="xg_module_body"> \
                            <p>' + args.bodyHtml + '</p> \
                            <p class="buttongroup"> \
                                <input type="button" class="button" value="' + dojo.string.escape('html', args.okButtonText) + '" /> \
                            </p> \
                        </div> \
                    </div> \
                </div>');
        var dialog = dojo.html.createNodesFromText(dialogHtml)[0];
        this.showOverlay();
        document.body.appendChild(dialog);
        dojo.event.connect(dojo.html.getElementsByClass('button', dialog)[0], 'onclick', dojo.lang.hitch(this, function(event) {
            dojo.event.browser.stopEvent(event);
            dojo.dom.removeNode(dialog);
            this.hideOverlay();
        }));
        return dialog;
    },

    /**
     * Displays a message indicating that processing is underway.
     *
     * @param args  object with the following properties:
     *         title     (optional) plain-text title, defaults to none
     *         bodyHtml  HTML for the dialog message (to be placed in a <p>)
     * @return  an object that you can call hide() on to hide the progress dialog
     */
    progressDialog: function(args) {
        if (dojo.byId('xg_lightbox_alert')) {
            dojo.dom.removeNode(dojo.byId('xg_lightbox_alert'));
        }
        var dialogHtml = dojo.string.trim(' \
                <div class="xg_floating_module" id="xg_lightbox_alert"> \
                    <div class="xg_floating_container"> \
                        <div class="xg_module_head ' + (args.title ? '' : 'notitle') + '"> \
                            ' + (args.title ? '<h2>' + dojo.string.escape('html', args.title) + '</h2>' : '') + ' \
                        </div> \
                        <div class="xg_module_body"> \
                            <img src="/xn_resources/widgets/index/gfx/spinner.gif" alt="" class="left" style="margin-right:5px" width="20" height="20"/> \
                            <p style="margin-left:25px">' + args.bodyHtml + '</p> \
                        </div> \
                    </div> \
                </div>');
        var dialog = dojo.html.createNodesFromText(dialogHtml)[0];
        this.showOverlay();
        document.body.appendChild(dialog);
        return { hide: dojo.lang.hitch(this, function() {
            dojo.dom.removeNode(dialog);
            this.hideOverlay();
        })};
    },

    /**
     * Displays a confirmation prompt.
     *
     * @param title  plain-text title; defaults to "Confirmation"
     * @param bodyText  plain text for the dialog message
     * @param bodyHtml  HTML for the dialog message; alternative to bodyText
     * @param okButtonText  text for the OK button; defaults to "OK"
     * @param onOk  function to call if the person presses OK. The dialog node is passed as the first argument.
     * @param closeOnlyIfOnOk bool if true, pressing OK closes the dialog only if onOk returns true
     */
    confirm: function(args) {
        args.title = args.title ? args.title : xg.shared.nls.text('confirmation');
        args.okButtonText = args.okButtonText ? args.okButtonText : xg.shared.nls.text('ok');
        args.onOk = args.onOk ? args.onOk : function() {};
        if (args.bodyText) { args.bodyHtml = '<p>' + dojo.string.escape('html', args.bodyText) + '</p>'; }
        // ContactList.js assumes that the dialog contains a <form>  [Jon Aquino 2007-10-25]
        var dialog = dojo.html.createNodesFromText(dojo.string.trim('\
                <div class="xg_floating_module"> \
                    <div class="xg_floating_container"> \
                        <div class="xg_module_head"> \
                            <h2>' + dojo.string.escape('html', args.title) + '</h2> \
                        </div> \
                        <div class="xg_module_body"> \
                            <form> \
                                ' + args.bodyHtml + ' \
                                <p class="buttongroup"> \
                                    <input type="submit" class="button" value="' + dojo.string.escape('html', args.okButtonText) + '" style="font-weight: bold" /> \
                                    <input type="button" class="button" value="' + xg.shared.nls.html('cancel') + '" /> \
                                </p> \
                            </form> \
                        </div> \
                    </div> \
                </div>'))[0];
        this.showOverlay();
        document.body.appendChild(dialog);
        dojo.event.connect(dojo.html.getElementsByClass('button', dialog)[1], 'onclick', dojo.lang.hitch(this, function(event) {
            dojo.event.browser.stopEvent(event);
            dojo.dom.removeNode(dialog);
            this.hideOverlay();
        }));
        dojo.event.connect(dialog.getElementsByTagName('form')[0], 'onsubmit', dojo.lang.hitch(this, function(event) {
            dojo.event.browser.stopEvent(event);
            // Hide dialog rather than remove it; otherwise radio buttons will be reset in IE (BAZ-) [Jon Aquino 2007-05-09]
            if (args.closeOnlyIfOnOk) {
                if (args.onOk(dialog)) {
                    dojo.style.hide(dialog);
                    xg.shared.util.hideOverlay();
                }
            } else {
                dojo.style.hide(dialog);
                xg.shared.util.hideOverlay();
                args.onOk(dialog);
            }
        }));
		return dialog;
    },

    /**
     * Prompts the user to join the network or current group. Once the user chooses to join, subsequent
     * calls to this function will simply call onOk().
     *
     * @param name  the text for the prompt, or null or an empty string to skip the prompt
     * @param membershipPending  whether the user's membership is pending approval
     * @param onOk  function to call if the person presses OK
     */
    promptToJoin: function(text, membershipPending, onOk) {
        if (typeof membershipPending == 'function') {
            onOk = membershipPending;
            membershipPending = false;
        }
        if (membershipPending) {
            this.promptIsPending();
            return;
        }
        if (this.joined || ! text) {
            onOk();
            return;
        }
        xg.shared.util.confirm({
            title: xg.shared.nls.text('joinNow'),
            bodyHtml: '<p>' + dojo.string.escape('html', text) + '</p>',
            okButtonText: xg.shared.nls.text('join'),
            onOk: dojo.lang.hitch(this, function() {
                this.joined = true;
                onOk();
            })
        });
    },

    /**
     * Tells the user that they can't continue since their membership is pending
     */
     promptIsPending: function() {
         xg.shared.util.alert({
                 title: xg.shared.nls.text('pendingPromptTitle'),
                 bodyHtml: '<p>' + xg.shared.nls.html('youCanDoThis') + '</p>'
         });
     },

    /**
     * Selects the contents of the given text field if you click it; makes it read-only.
     * Useful for text fields containing <embed> code.
     *
     * @param input  a text field
     */
    selectOnClick: function(input) {
        dojo.event.connect(input, 'onfocus', function(event) {
            dojo.html.selectInputText(input);
        });
        dojo.event.connect(input, 'onclick', function(event) {
            dojo.html.selectInputText(input);
        });
        var text = input.value;
        dojo.event.connect(input, 'onkeyup', function(event) {
            dojo.html.selectInputText(input);
            input.value = text;
        });
    },

    /**
     * Displays the form for editing the properties of a module
     * on, for example, the homepage.
     *
     * @param form  the <form> node
     * @param formHeight  the original height of the form, in pixels
     * @param editButton  the Edit <a>
     */
    showModuleForm: function(form, formHeight, editButton) {
        dojo.html.addClass(editButton, 'close');
        var animation1 = new dojo.animation.Animation(new dojo.math.curves.Line( [0,0], [0,formHeight+5]), 300, .8, 0);
        var animation2 = new dojo.animation.Animation(new dojo.math.curves.Line( [0,formHeight+5], [0,formHeight]), 20, -.8, 0);
        dojo.event.connect(animation1, "onAnimate", function(e) { form.style.height = (e.y) + "px"; });
        dojo.event.connect(animation2, "onAnimate", function(e) { form.style.height = (e.y) + "px"; });
        dojo.event.connect(animation1, "onEnd", function(e) { animation2.play(); });
        animation1.play();
    },

    /**
     * Hides the form for editing the properties of a module
     * on, for example, the homepage.
     *
     * @param form  the <form> node
     * @param formHeight  the original height of the form, in pixels
     * @param editButton  the Edit <a>
     */
    hideModuleForm: function(form, formHeight, editButton) {
        dojo.html.removeClass(editButton, 'close');
        var animation1 = new dojo.animation.Animation(new dojo.math.curves.Line( [0,formHeight],[0,formHeight+5]), 20, .8, 0);
        var animation2 = new dojo.animation.Animation(new dojo.math.curves.Line( [0,formHeight+5], [0,0]), 300, -.8, 0);
        dojo.event.connect(animation1, "onAnimate", function(e) { form.style.height = (e.y) + "px"; });
        dojo.event.connect(animation1, "onEnd", function(e) { animation2.play(); });
        dojo.event.connect(animation2, "onAnimate", function(e) { form.style.height = (e.y) + "px"; });
        dojo.event.connect(animation2, "onEnd", function(e) { dojo.html.addClass(form, 'collapsed'); });
        animation1.play();
    },

    /**
     * Intercepts Enter keypresses in the textfield so that its containing form is not submitted.
     *
     * @param textInput  the text <input>
     * @param onEnterPress  optional callback to call when Enter is pressed
     */
    preventEnterFromSubmittingForm: function(textInput, onEnterPress) {
        if (! onEnterPress) { onEnterPress = function() {}; }
        dojo.event.connect(textInput, 'onkeydown', function(event) {
            if (event.keyCode == dojo.event.browser.keys.KEY_ENTER) {
                dojo.event.browser.stopEvent(event);
                onEnterPress();
            }
        });
    },

    /**
     * Sets the default value for the textfield that is cleared upon focus or form submit.
     *
     * @param textInput  <input:text>
     * @param value  	string			Default value
     */
    setPlaceholder: function(textInput, value) {
        if (textInput.value != '') {
            return;
        }
        textInput.value = value;
        dojo.event.connect(textInput, 'onfocus', function(event) {
            if (textInput.value == value) {
                textInput.value = '';
            }
        });
        dojo.event.connect(textInput, 'onblur', function(event) {
            if (textInput.value == '') {
                textInput.value = value;
            }
        });
        dojo.event.connect(textInput.form, 'onsubmit', function(event) {
            if (textInput.value == value) {
                textInput.value = '';
            }
        });
    }
};