dojo.provide('xg.index.message.friendList');

dojo.require('dojo.json');
dojo.require('dojo.event.topic');

/**
 * Behavior for the friendList page.
 */
xg.index.message.friendList = {

    /** Whether to display user thumbnails, or remove them to improve performance. */
    showAvatars: null,

    /**
     * The number of friends that are currently selected.
     * Equal to numFriends when All Friends is chosen.
     */
    selectedFriendCount: 0,

    /** The number of friends with ticked checkboxes. */
    checkedFriendCount: 0,

    /**
     * Friends, keyed by screenName. Each friend is represented by an object
     * with thumbnailUrl, screenName, fullName, and checked.
     */
    friends: {},

    /** Friends, keyed by page number then by screenName. */
    friendsByPage: {},

    /** HTML for the pagination, keyed by page number. */
    paginationHtmlByPage: {},

    /** Endpoint for retrieving a page of friends. */
    friendDataUrl: null,

    /** The total number of friends of the current user. */
    numFriends: null,

    /** Whether this object is currently building a page for display. */
    buildingPage: false,

    /** The pagination <ul> node. */
    paginationUl: null,

    /** The div containing the "We're loading your friends" message. */
    loadingMessageDiv: null,

    /** The <ul> listing the friends. */
    friendUl: null,

    /** Whether the first page has been displayed. */
    firstPageDisplayed: false,

    /**
     * Sets up this object. May take a few seconds, so delay it as long as possible.
     */
    initialize: function() {
        this.friendUl = dojo.byId('friendlist').getElementsByTagName('ul')[0];
        this.showAvatars = this.friendUl.getAttribute('_showAvatars') !== 'false';
        this.friendDataUrl = this.friendUl.getAttribute('_friendDataUrl');
        this.numFriends = this.friendUl.getAttribute('_numFriends');
        this.loadingMessageDiv = dojo.byId('loading_message_container');
        this.setSelectedFriendCount(dojo.byId('all_friends_radio_button').checked ? this.numFriends : 0);
        dojo.event.connect(dojo.byId('all_friends_radio_button'), 'onclick', dojo.lang.hitch(this, function(event) {
            this.setSelectedFriendCount(this.numFriends);
            dojo.style.hide(dojo.byId('friend_count'));
            dojo.style.hide(dojo.byId('friendlist'));
            if (this.paginationUl) { dojo.style.hide(this.paginationUl); }
        }));
        dojo.event.connect(dojo.byId('these_friends_radio_button'), 'onclick', dojo.lang.hitch(this, function(event) {
            this.setSelectedFriendCount(this.checkedFriendCount);
            dojo.style.show(dojo.byId('friend_count'));
            dojo.style.show(dojo.byId('friendlist'));
            if (this.paginationUl) { dojo.style.show(this.paginationUl); }
            if (! this.firstPageDisplayed) {
                this.showPage(1);
            }
        }));
        dojo.style.insertCssRule('body.busy *', 'cursor: wait');
    },

    /**
     * Sets whether this object is currently building a page for display.
     *
     * @param buildingPage  whether to disable the controls
     */
    setBuildingPage: function(buildingPage) {
        this.buildingPage = buildingPage;
        if (! this.firstPageDisplayed) {
            if (buildingPage) {
                dojo.style.hide(this.friendUl);
                dojo.style.show(this.loadingMessageDiv);
            } else {
                dojo.style.show(this.friendUl);
                dojo.style.hide(this.loadingMessageDiv);
                this.firstPageDisplayed = true;
            }
        } else {
            // When moving between pages, just use an hourglass.
            // Switching to the loadingMessageDiv is too jarring. [Jon Aquino 2007-12-31]
            if (buildingPage) {
                dojo.html.addClass(document.body, 'busy');
            } else {
                dojo.html.removeClass(document.body, 'busy');
            }
        }
        if (! buildingPage) { dojo.event.topic.publish('xg.index.message.friendList.pageBuilt'); }
    },

    /**
     * Notifies this object that the form is about to be submitted.
     */
    onSubmit: function() {
        if (! dojo.byId('these_friends_radio_button').checked) { return; }
        var checkedScreenNames = [];
        for (screenName in this.friends) {
            var friend = this.friends[screenName];
            if (typeof friend != 'object') { continue; }  // Just in case [Jon Aquino 2007-12-28]
            if (friend.checked) { checkedScreenNames.push(friend.screenName); }
        }
        var form = dojo.dom.getFirstAncestorByTag(this.friendUl, 'form');
        form.screenNames.value = dojo.json.serialize(checkedScreenNames);
    },

    /**
     * Adds event handlers to the pagination.
     *
     * @param paginationUl  the pagination <ul> node
     */
    initializePagination: function(paginationUl) {
        dojo.lang.forEach(paginationUl.getElementsByTagName('a'), dojo.lang.hitch(this, function(a) {
            dojo.event.connect(a, 'onclick', dojo.lang.hitch(this, function(event) {
                dojo.event.browser.stopEvent(event);
                this.showPage(a.innerHTML);
            }));
        }));
    },

    /**
     * Displays the given page.
     *
     * @param page  the page number
     */
    showPage: function(page) {
        if (this.buildingPage) { return; }
        this.setBuildingPage(true);
        this.friendsOnPage(page, dojo.lang.hitch(this, function(friendsOnPage, paginationHtml) {
            var listItems = [];
            dojo.lang.forEach(friendsOnPage, dojo.lang.hitch(this, function(friend) {
                    // Array.join is 25x faster than + in IE [Jon Aquino 2007-12-29]
                    listItems.push(['<li', friend.reasonToDisable ? ' class="member"' : '', '><label><input type="checkbox" class="checkbox" onclick="xg.index.message.friendList.checkboxClicked(this, \'', friend.screenName, '\')"', friend.checked ? ' checked="checked"' : '', friend.reasonToDisable ? ' disabled="disabled"' : '', ' /> ', this.showAvatars ? '' : '<!--', '<img src="', friend.thumbnailUrl, '" width="32" height="32" alt="" />', this.showAvatars ? '' : '-->', ' <span class="name">', dojo.string.escape('html', friend.fullName), friend.reasonToDisable ? ' <small>' + friend.reasonToDisable + '</small>' : '', '</span></label></li>'].join(''));
            }));
            this.friendUl.innerHTML = listItems.join('');
            if (this.paginationUl) { dojo.dom.removeNode(this.paginationUl); }
            if (paginationHtml.length > 0) {
                var container = document.createElement('div');
                container.innerHTML = paginationHtml;
                this.paginationUl = container.getElementsByTagName('ul')[0];
                dojo.html.setClass(this.paginationUl, 'pagination smallpagination easyclear');
                dojo.dom.insertAfter(this.paginationUl, dojo.byId('friendlist'));
                this.initializePagination(this.paginationUl);
            }
            this.setBuildingPage(false);
        }));
    },

    /**
     * Called when the user selects a checkbox.
     *
     * @param checkbox  the checkbox input element
     * @param screenName  the screen name of the associated user
     */
    checkboxClicked: function(checkbox, screenName) {
        this.friends[screenName].checked = checkbox.checked;
        this.checkedFriendCount = Math.max(0, this.checkedFriendCount + (checkbox.checked ? 1 : -1));
        this.setSelectedFriendCount(this.checkedFriendCount);
        dojo.byId('these_friends_radio_button').checked = true;
    },

    /**
     * Sets the number of friends that are currently selected.
     *
     * @param selectedFriendCount  the number of friends that the user has chosen
     */
    setSelectedFriendCount: function(selectedFriendCount) {
        this.selectedFriendCount = selectedFriendCount;
        dojo.byId('friend_count').innerHTML = xg.index.nls.html('nFriends', this.selectedFriendCount);
    },


    /**
     * Retrieves the friends on the given page.
     *
     * @param page  the page number
     * @param callback  function to call with the array of friends and the pagination HTML
     */
    friendsOnPage: function(page, callback) {
        if (this.friendsByPage[page] !== undefined) {
            callback.call(this, this.friendsByPage[page], this.paginationHtmlByPage[page]);
        } else {
            dojo.io.bind({
                url: this.friendDataUrl + '&page=' + page,
                method: 'post',
                preventCache: true,
                encoding: 'utf-8',
                mimetype: 'text/javascript',
                load: dojo.lang.hitch(this, function(type, data, event) {
                    this.friendsByPage[page] = dojo.json.evalJson(data.friends);
                    this.paginationHtmlByPage[page] = data.paginationHtml;
                    dojo.lang.forEach(this.friendsByPage[page], dojo.lang.hitch(this, function(friend) {
                        this.friends[friend.screenName] = friend;
                    }));
                    callback.call(this, this.friendsByPage[page], this.paginationHtmlByPage[page]);
                })
            });
        }
    }

};


