<?php

/**
 * Dispatches requests pertaining to discussion topics.
 */
class Forum_TopicController extends XG_GroupEnabledController {

    /**
     * Pokes a hole in the app-wide privacy mechanism.
     *
     * @param $action string  The name of the action
     * @return boolean  Whether to bypass the privacy mechanism for the given action
     */
    public function action_overridePrivacy($action) {
        return ! XG_App::appIsPrivate() && ! XG_GroupHelper::groupIsPrivate() && $_GET['feed'] == 'yes' && in_array($action, array('show', 'search', 'list', 'listForCategory', 'listForTag', 'listForContributor'));
    }

    /**
     * Displays the form for a new discussion topic.
     *
     * Expected GET variables:
     *     categoryId - ID of the initial category (optional)
     */
    public function action_new($errors = NULL) {
        XG_SecurityHelper::redirectIfNotMember();
        XG_JoinPromptHelper::joinGroupOnSave();
        XG_App::includeFileOnce('/lib/XG_TagHelper.php');
        $this->_widget->includeFileOnce('/lib/helpers/Forum_FileHelper.php');
        $this->_widget->includeFileOnce('/lib/helpers/Forum_SecurityHelper.php');
        if (! Forum_SecurityHelper::currentUserCanAddTopic()) { throw new Exception('Cannot add topics (1534327525)'); }
        $this->errors = $errors;
        $this->categories = Category::findAll(XG_SecurityHelper::userIsAdmin() || XG_GroupHelper::isGroupAdmin());
        $this->form = new XNC_Form(array('categoryId' => $_GET['categoryId']));
        $this->title = xg_text('START_A_NEW_DISCUSSION');
        $this->buttonText = xg_text('START_DISCUSSION');
        $this->formUrl = $this->_buildUrl('topic', 'create');
        $this->emptyAttachmentSlotCount = Topic::emptyAttachmentSlotCount(Topic::create());
        $this->hideStartDiscussionLink = true;
        $this->render('newOrEdit');
    }

    /**
     * Processes the form for a new discussion topic.
     */
    public function action_create() {
        XG_SecurityHelper::redirectIfNotMember();
        XG_JoinPromptHelper::joinGroupOnSave();
        if ($_SERVER['REQUEST_METHOD'] != 'POST') { $this->redirectTo('new', 'topic'); return; }
        $this->_widget->includeFileOnce('/lib/helpers/Forum_SecurityHelper.php');
        W_Cache::getWidget('main')->includeFileOnce('/lib/helpers/Index_NotificationHelper.php');
        if (! Forum_SecurityHelper::currentUserCanAddTopic()) { throw new Exception('Cannot add topics'); }
        $this->processForm($topic = Topic::create(), 'new');
        // BAZ-4128: Subscribe the creator to the notification alias unless he
        //   has opted out of notifications for comments on his content
        $user = User::load(XN_Profile::current());
        if ($user && $user->my->emailActivityPref !== 'N') {
            Index_NotificationHelper::startFollowing($topic);
        }
        Category::invalidateRecentTopicsCache(Category::find($topic->my->categoryId));
        $this->_widget->includeFileOnce('/lib/helpers/Forum_NotificationHelper.php');
        Forum_NotificationHelper::notifyNewTopicFollowers($topic);
    }

    /**
     * Sets the tags for the given object for the current user.
     *
     * Expected GET parameters:
     *     xn_out  Should always be "json"
     */
    public function action_tag() {
        $x = $this->_buildPath('lib/helpers/Forum_SecurityHelper.php');  include_once $x;
        XG_App::includeFileOnce('/lib/XG_TagHelper.php');
        $x = $this->_buildPath('lib/helpers/Forum_HtmlHelper.php');  include_once $x;
        if (! Forum_SecurityHelper::currentUserCanTag()) { throw new Exception('Not allowed (1928669282)'); }
        if ($_SERVER['REQUEST_METHOD'] != 'POST') { throw new Exception('Not a POST (498350803)'); }
        XG_HttpHelper::trimGetAndPostValues();
        $topic = XG_GroupHelper::checkCurrentUserCanAccess(W_Content::load($_GET['id']));
        if ($topic->type != 'Topic') { throw new Exception('Not a Topic'); }
        XG_TagHelper::setTagStringForObjectAndCurrentUser($topic, mb_substr($_POST['tags'], 0, XG_TagHelper::MAX_TAGS_LENGTH));
        XG_TagHelper::updateTopTags($topic, Topic::TOP_TAGS_COUNT);
        $this->html = Forum_HtmlHelper::tagHtmlForDetailPage(XG_TagHelper::getTagNamesForObject($topic));
    }

    /**
     * Displays the detail page for a discussion topic.
     *
     * Expected GET variables:
     *     id - ID of the Topic object
     *     page - page number (optional)
     */
    public function action_show() {
        // If the user clicked an 'unfollow' link, make sure he's logged in,
        //   otherwise we can't remove him from the follow list
        if ($_GET['unfollow']) {
            XG_SecurityHelper::redirectIfNotMember();
            $this->unFollow = true;
        }
        XG_App::includeFileOnce('/lib/XG_TagHelper.php');
        XG_App::includeFileOnce('/lib/XG_ContactHelper.php');
        XG_App::includeFileOnce('/lib/XG_PaginationHelper.php');
        $this->_widget->includeFileOnce('/lib/helpers/Forum_HtmlHelper.php');
        $this->_widget->includeFileOnce('/lib/helpers/Forum_SecurityHelper.php');
        $this->_widget->includeFileOnce('/lib/helpers/Forum_FileHelper.php');
        $this->_widget->includeFileOnce('/lib/helpers/Forum_CommentHelper.php');
        W_Cache::getWidget('main')->includeFileOnce('/lib/helpers/Index_NotificationHelper.php');
        try {
            $this->topic = W_Content::load($_GET['id']);
        } catch (Exception $e) {
            W_Cache::getWidget('main')->dispatch('error','404');
            exit;
        }
        XG_GroupHelper::checkCurrentUserCanAccess($this->topic);
        if ($this->_widget->privateConfig['allCategoriesDeletedOn'] && strtotime($this->topic->createdDate) < strtotime($this->_widget->privateConfig['allCategoriesDeletedOn'])) {
            $this->render('showDeleted');
            return;
        }
        if ($_GET['feed'] == 'yes') {
            header('Content-Type: application/atom+xml');
            XG_App::includeFileOnce('/lib/XG_FeedHelper.php');
            XG_FeedHelper::cacheFeed(array('id' => 'forum-topic-show-' . md5(XG_HttpHelper::currentUrl())));
            XG_FeedHelper::outputFeed(Forum_CommentHelper::createQuery($this->topic->id, 0, 10, 'mostRecent')->execute(), $this->topic->title, NULL);
            exit;
        }
        // Was there an unfollow GET parameter from link in follow notification email?
        if ($this->unFollow) {
            Index_NotificationHelper::stopFollowing($this->topic);
        }
        $this->currentCommentId = $_GET['commentId'];
        // Not too many, otherwise avatar loading slows down page [Jon Aquino 2007-01-30]
        $this->pageSize = Forum_CommentHelper::COMMENTS_PER_PAGE;
        $begin = XG_PaginationHelper::computeStart($_GET['page'], $this->pageSize);
        $query = Forum_CommentHelper::createQuery($this->topic->id, $begin, $begin + $this->pageSize
                + 1); // Query has one extra result so we can determine if each comment has child comments [Jon Aquino 2007-04-04]
        $commentsPlusOne = $query->execute();
        $this->comments = array_slice($commentsPlusOne, 0, $this->pageSize);
        if ($this->currentCommentId && ! in_array($this->currentCommentId, Forum_CommentHelper::ids($this->comments))) {
            try {
                $comment = XG_Cache::content($this->currentCommentId);
            } catch (Exception $e) {
            }
            if (! $comment) {
                $this->redirectTo('showDeleted', 'comment', '?topicId=' . $this->topic->id);
                return;
            }
            $components = parse_url($url = Forum_CommentHelper::url($comment));
            parse_str($components['query'], $parameters);
            // Check page, just in case, to prevent infinite redirection loop [Jon Aquino 2007-04-05]
            if (XG_PaginationHelper::computeStart($parameters['page'], $this->pageSize) != XG_PaginationHelper::computeStart($_GET['page'], $this->pageSize)) {
                header('Location: ' . $url);
                exit;
            }
        }
        $this->commentIdsWithChildComments = Forum_CommentHelper::commentIdsWithChildComments($this->pageSize, $commentsPlusOne);
        $this->category = Category::find($this->topic->my->categoryId);
        if ($this->category) { $this->categoryUrl = XG_GroupHelper::buildUrl($this->_widget->dir, 'topic', 'listForCategory', array('categoryId' => $this->category->id)); }
        if (count(Category::findAll()) && ! $this->category) {
            $this->render('showDeleted');
            return;
        }
        $this->tags = XG_TagHelper::getTagNamesForObject($this->topic);
        $this->feedUrl = XG_HttpHelper::addParameters(XG_HttpHelper::currentUrl(), array('feed' => 'yes', 'xn_auth' => 'no'));
        $this->feedDescription = xg_text('SUBSCRIBE_TO_DISCUSSION');
        $this->totalCount = $query->getTotalCount();
        if (count($this->comments) == 0 && $_GET['page'] > 0) {
            // Deleted the last comment on the current page [Jon Aquino 2007-01-25]
            $this->redirectTo('show', 'topic', array('id' => $this->topic->id));
            return;
        }
        XG_App::includeFileOnce('/lib/XG_MetatagHelper.php');
        $this->metaDescription = XG_MetatagHelper::forDescription($this->topic->description);
        $this->metaKeywords = false;
        if (sizeof($this->tags)) {
            $metaKeywords =  array();
            foreach($this->tags as $tag) {
                $metaKeywords[] = xnhtmlentities($tag);
            }
            $this->metaKeywords = implode(', ',$metaKeywords);
        }
        $this->showDiscussionClosedModule = $_GET['repliedToClosedDiscussion'];
        $currentUserTags = $this->_user->isLoggedIn() ? XG_TagHelper::getTagsForObjectAndUser($this->topic->id, $this->_user->screenName) : array();
        $this->currentUserTagString = XG_TagHelper::implode(XN_Tag::tagNamesFromTags($currentUserTags));
        $categories = Category::findAll(XG_SecurityHelper::userIsAdmin() || XG_GroupHelper::isGroupAdmin());
        if (Forum_SecurityHelper::currentUserCanSetCategory($this->topic) && count($categories) > 1) {
            $this->categoryPickerOptionsJson = $this->categoryPickerOptionsJson($categories, $this->category);
        }
        XG_Cache::profiles($this->topic, $this->comments);
    }

    /**
     * Displays an error page saying that the discussion has been deleted.
     *
     * @param $showNavigation boolean  whether to show the navigation links
     */
    public function action_showDeleted($showNavigation = true) {
        $this->showNavigation = $showNavigation;
    }


    /**
     * Redirects to the page showing the last reply for the given topic
     *
     * Expected GET variables:
     *     id - ID of the Topic object
     */
    public function action_showLastReply() {
        $this->_widget->includeFileOnce('/lib/helpers/Forum_CommentHelper.php');
        $results = XN_Query::create('Content')->filter('owner')->filter('type', '=', 'Comment')->filter('my->attachedTo', '=', $_GET['id'])->order('createdDate', 'desc', XN_Attribute::DATE)->end(1)->execute();
        $url = $results[0] ? Forum_CommentHelper::urlProper($results[0]) : $this->_buildUrl('topic', 'show', array('id' => $_GET['id']));
        header('Location: ' . $url);
    }

    /**
     * Displays the form for editing a discussion topic.
     *
     * Expected GET variables:
     *     id - ID of the Topic to edit
     */
    public function action_edit($errors = NULL) {
        XG_SecurityHelper::redirectIfNotMember();
        XG_App::includeFileOnce('/lib/XG_TagHelper.php');
        $this->_widget->includeFileOnce('/lib/helpers/Forum_FileHelper.php');
        $this->_widget->includeFileOnce('/lib/helpers/Forum_SecurityHelper.php');
        $topic = XG_GroupHelper::checkCurrentUserCanAccess(W_Content::load($_GET['id']));
        if ($topic->type != 'Topic') { throw new Exception('Not a Topic'); }
        if (! Forum_SecurityHelper::currentUserCanEditTopic($topic)) { throw new Exception('Cannot edit topic (1277898673)'); }
        $this->errors = $errors;
        $tags = XG_TagHelper::getTagsForObjectAndUser($topic->id, $this->_user->screenName);
        $tagString = XG_TagHelper::implode(XN_Tag::tagNamesFromTags($tags));
        $this->categories = Category::findAll(XG_SecurityHelper::userIsAdmin() || XG_GroupHelper::isGroupAdmin());
        $category = Category::find($topic->my->categoryId);
        $this->form = new XNC_Form(array('title' => $topic->title, 'description' => $topic->description, 'tags' => $tagString, 'categoryId' => $category ? $category->id : null));
        $this->title = xg_text('EDIT_DISCUSSION');
        $this->buttonText = xg_text('SAVE_CHANGES');
        $this->formUrl = $this->_buildUrl('topic', 'update', array('id' => $topic->id));
        $this->emptyAttachmentSlotCount = Topic::emptyAttachmentSlotCount($topic);
        $this->render('newOrEdit');
    }

    /**
     * Processes the form for editing a discussion topic.
     *
     * Expected GET variables:
     *     id - ID of the Topic to edit
     */
    public function action_update() {
        XG_SecurityHelper::redirectIfNotMember();
        XG_JoinPromptHelper::joinGroupOnSave();
        $this->_widget->includeFileOnce('/lib/helpers/Forum_SecurityHelper.php');
        $topic = XG_GroupHelper::checkCurrentUserCanAccess(W_Content::load($_GET['id']));
        if ($_SERVER['REQUEST_METHOD'] != 'POST') { $this->redirectTo('edit', 'topic', array('id' => $topic->id)); return; }
        if ($topic->type != 'Topic') { throw new Exception('Not a Topic'); }
        if (! Forum_SecurityHelper::currentUserCanEditTopic($topic)) { throw new Exception('Cannot edit topic (502036165)'); }
        $this->processForm($topic, 'edit');
    }

    /**
     * Updates the given Topic object using the posted form variables, then
     * forwards or redirects to an appropriate page.
     * Redirects to the sign-up page if the person is signed out.
     *
     * @param $topic W_Content  The Topic to update
     * @param $actionOnError string  Action to forward to if an error occurs
     */
    private function processForm(W_Content $topic, $actionOnError) {
        XG_App::includeFileOnce('/lib/XG_FileHelper.php');
        XG_App::includeFileOnce('/lib/XG_HttpHelper.php');
        XG_App::includeFileOnce('/lib/XG_TagHelper.php');
        $this->_widget->includeFileOnce('/lib/helpers/Forum_SecurityHelper.php');
        $this->_widget->includeFileOnce('/lib/helpers/Forum_HtmlHelper.php');
        $this->_widget->includeFileOnce('/lib/helpers/Forum_FileHelper.php');
        $this->_widget->includeFileOnce('/lib/helpers/Forum_UserHelper.php');
        if ($_POST['categoryId'] && ! Forum_SecurityHelper::currentUserCanAddTopicToCategory(Category::find($_POST['categoryId']))) { xg_echo_and_throw('Not allowed (1650948551)'); }
        XG_HttpHelper::trimGetAndPostValues();
        $errors = array();
        if (! $_POST['title']) { $errors['title'] = xg_text('PLEASE_ENTER_TITLE'); }
        if (! $_POST['description']) { $errors['description'] = xg_text('PLEASE_ENTER_FIRST_POST'); }
        for ($i = 1; $i <= Topic::emptyAttachmentSlotCount($topic); $i++) {
            if ($_POST["file$i"] && $_POST["file$i:status"]) { $errors["file$i"] = XG_FileHelper::uploadErrorMessage($_POST["file$i:status"]); }
        }
        if (count($errors)) {
            $this->forwardTo($actionOnError, 'topic', array($errors));
            return;
        }
        $topic->title = Topic::cleanTitle($_POST['title']);
        $topic->description = Topic::cleanDescription($_POST['description']);
        if ($_POST['file1']) { Forum_FileHelper::addAttachment('file1', $topic); }
        if ($_POST['file2']) { Forum_FileHelper::addAttachment('file2', $topic); }
        if ($_POST['file3']) { Forum_FileHelper::addAttachment('file3', $topic); }
        Category::setCategoryId($topic, $_POST['categoryId'] ? $_POST['categoryId'] : null);
        $topic->save();
        if (! $topic->my->lastEntryDate) {
            $topic->my->set('lastEntryDate', $topic->createdDate, XN_Attribute::DATE);
            $topic->save();
        }
        Forum_UserHelper::updateActivityCount(User::load($this->_user));
        XG_TagHelper::setTagStringForObjectAndCurrentUser($topic, mb_substr($_POST['tags'], 0, XG_TagHelper::MAX_TAGS_LENGTH));
        XG_TagHelper::updateTopTags($topic, Topic::TOP_TAGS_COUNT);
        if($actionOnError == 'new'){
            XG_App::includeFileOnce('/lib/XG_ActivityHelper.php');
            if ($topic->my->groupId) {
                XG_ActivityHelper::logActivityIfEnabled(XG_ActivityHelper::CATEGORY_NEW_CONTENT, XG_ActivityHelper::SUBCATEGORY_GROUP_TOPIC, $topic->contributorName, array($topic,XG_GroupHelper::currentGroup()));
            } else {
                XG_ActivityHelper::logActivityIfEnabled(XG_ActivityHelper::CATEGORY_NEW_CONTENT, XG_ActivityHelper::SUBCATEGORY_TOPIC, $topic->contributorName, array($topic));
            }
        }
        if (XG_GroupHelper::inGroupContext()) {
            $group = XG_GroupHelper::currentGroup();
            Group::updateActivityScore($group,GROUP::ACTIVITY_SCORE_FORUM_TOPIC);
            $group->save();
        }
        $this->redirectTo('show', 'topic', array('id' => $topic->id));
    }

    /**
     * Displays a list of recent discussion topics.
     *
     * Expected GET variables:
     *     page - page number (optional)
     *     sort - mostRecentDiscussions, mostPopularDiscussions, or mostRecentlyUpdatedDiscussions
     */
    public function action_list() {
        $this->_widget->includeFileOnce('/lib/helpers/Forum_SecurityHelper.php');
        $this->showSidebar = ! XG_GroupHelper::inGroupContext();
        self::prepareListAction(array(
                'titleHtml' => XG_GroupHelper::inGroupContext() ? xg_html('GROUP_FORUM', xnhtmlentities(XG_GroupHelper::currentGroup()->title)) : xg_html('FORUM'),
                'templateForNoDiscussions' => Forum_SecurityHelper::currentUserCanSeeAddTopicLinks() ? 'listEmptyWithAddButton' : 'listEmpty',
                'sortNames' => array('mostRecentlyUpdatedDiscussions', 'mostRecentDiscussions', 'mostPopularDiscussions')));
        $this->showListForContributorLinks = FALSE;
        $this->showContributorName = TRUE;
        $this->feedDescription = xg_text('SUBSCRIBE_TO_DISCUSSIONS');
        $this->noDiscussionsHtml = Forum_SecurityHelper::currentUserCanSeeAddTopicLinks() ? xg_html('NOBODY_HAS_ADDED_DISCUSSIONS_ADD') : xg_html('NOBODY_HAS_ADDED_DISCUSSIONS');
        $this->showFollowLink = Forum_SecurityHelper::currentUserCanFollowNewTopics();
    }

    /**
     * Displays a list of recent discussion topics in a given category.
     *
     * Expected GET variables:
     *     categoryId - the ID of the Category
     *     page - page number (optional)
     *     sort - mostRecentDiscussions, mostPopularDiscussions, or mostRecentlyUpdatedDiscussions
     */
    public function action_listForCategory() {
        $this->_widget->includeFileOnce('/lib/helpers/Forum_SecurityHelper.php');
        $_GET['categoryId'] = urldecode($_GET['categoryId']); // Workaround for BAZ-4133 [Jon Aquino 2007-09-27]
        $category = Category::find($_GET['categoryId']);
        if (! $category) {
            W_Cache::getWidget('main')->dispatch('error','404');
            exit;
        }
        self::prepareListAction(array(
                'titleHtml' => xnhtmlentities($category->title),
                'templateForNoDiscussions' => Forum_SecurityHelper::currentUserCanSeeAddTopicLinks() ? 'listEmptyWithAddButton' : 'listEmpty',
                'sortNames' => array('mostRecentlyUpdatedDiscussions', 'mostRecentDiscussions', 'mostPopularDiscussions'),
                'category' => $category));
        $this->showListForContributorLinks = FALSE;
        $this->showContributorName = TRUE;
        $this->feedDescription = xg_text('SUBSCRIBE_TO_DISCUSSIONS');
        $this->noDiscussionsHtml = Forum_SecurityHelper::currentUserCanSeeAddTopicLinks() ? xg_html('NOBODY_HAS_ADDED_DISCUSSIONS_ADD') : xg_html('NOBODY_HAS_ADDED_DISCUSSIONS');
        if ($category->description && $_GET['page'] < 2) {
            $this->description = $category->description;
        }
    }

    /**
     * Displays a list of recent discussion topics with the given search keywords.
     *
     * Expected GET variables:
     *     page - page number (optional)
     *     sort - mostRecent or mostPopularDiscussions
     *     q - search keywords (optional)
     *
     */
    public function action_search() {
        XG_App::includeFileOnce('/lib/XG_QueryHelper.php');
        self::prepareListAction(array(
                'titleHtml' => xg_html('DISCUSSIONS_MATCHING_X', xnhtmlentities($_GET['q'])),
                'templateForNoDiscussions' => 'listEmptyForSearch',
                'useSearch' => (XG_QueryHelper::getSearchMethod() != 'content'),
                'sortNames' => array('mostRecent', 'mostPopularDiscussions')));
        $this->showListForContributorLinks = FALSE;
        $this->showContributorName = TRUE;
        $this->feedDescription = xg_text('SUBSCRIBE_TO_DISCUSSIONS');
        $this->noDiscussionsHtml = xg_html('WE_COULD_NOT_FIND_ANY_DISCUSSIONS');
        $this->hideSorts = true;
    }

    /**
     * Displays a list of recent discussion topics with a given tag
     *
     * Expected GET variables:
     *     page - page number (optional)
     *     sort - mostRecentDiscussions, mostPopularDiscussions, or mostRecentlyUpdatedDiscussions
     *     tag - the tag
     */
    public function action_listForTag() {
        $this->_widget->includeFileOnce('/lib/helpers/Forum_SecurityHelper.php');
        self::prepareListAction(array(
                'titleHtml' => xg_html('ALL_DISCUSSIONS_TAGGED_X', xnhtmlentities($_GET['tag'])),
                'templateForNoDiscussions' => Forum_SecurityHelper::currentUserCanSeeAddTopicLinks() ? 'listEmptyWithAddButton' : 'listEmpty',
                'sortNames' => array('mostRecentlyUpdatedDiscussions', 'mostRecentDiscussions', 'mostPopularDiscussions')));
        $this->showListForContributorLinks = FALSE;
        $this->showContributorName = TRUE;
        $this->feedDescription = xg_text('SUBSCRIBE_TO_DISCUSSIONS_TAGGED_X', $_GET['tag']);
        $this->noDiscussionsHtml = Forum_SecurityHelper::currentUserCanSeeAddTopicLinks()
            ? xg_html('NO_DISCUSSIONS_TAGGED_X_CHECK_ADD', xnhtmlentities($_GET['tag']), 'href="' . xnhtmlentities(W_Cache::getWidget('photo')->buildUrl('photo', 'listTagged', array('tag' => $_GET['tag']))) . '"', 'href="' . xnhtmlentities(W_Cache::getWidget('video')->buildUrl('video', 'listTagged', array('tag' => $_GET['tag']))) . '"', 'href="' . xnhtmlentities(W_Cache::getWidget('profiles')->buildUrl('blog', 'list', array('tag' => $_GET['tag']))) . '"')
            : xg_html('NO_DISCUSSIONS_TAGGED_X_CHECK', xnhtmlentities($_GET['tag']), 'href="' . xnhtmlentities(W_Cache::getWidget('photo')->buildUrl('photo', 'listTagged', array('tag' => $_GET['tag']))) . '"', 'href="' . xnhtmlentities(W_Cache::getWidget('video')->buildUrl('video', 'listTagged', array('tag' => $_GET['tag']))) . '"', 'href="' . xnhtmlentities(W_Cache::getWidget('profiles')->buildUrl('blog', 'list', array('tag' => $_GET['tag']))) . '"');
        $this->showNoDiscussionsHeading = FALSE;
    }

    /**
     * Displays a list of recent discussion topics started by a given person.
     *
     * Expected GET variables:
     *     page - page number (optional)
     *     sort - mostRecent or mostPopularDiscussions
     *     user - screen name of the person who started the topics
     */
    public function action_listForContributor() {
        $this->_widget->includeFileOnce('/lib/helpers/Forum_SecurityHelper.php');
        if (! $_GET['user']) {
            XG_SecurityHelper::redirectIfNotMember();
            // Redirect; otherwise Bloglines bookmarklet will hit sign-in page when looking for RSS autodiscovery elements  [Jon Aquino 2007-01-19]
            $this->redirectTo('listForContributor', 'topic', array('user' => XN_Profile::current()->screenName));
            return;
        }
        $fullName = xg_username(XG_Cache::profiles($_GET['user']));
        self::prepareListAction(array(
                'titleHtml' => $_GET['user'] == XN_Profile::current()->screenName ? xg_html('MY_DISCUSSIONS') : xg_html('XS_DISCUSSIONS', xnhtmlentities($fullName)),
                'templateForNoDiscussions' => $_GET['user'] == XN_Profile::current()->screenName && Forum_SecurityHelper::currentUserCanSeeAddTopicLinks() ? 'listEmptyWithAddButton' : 'listEmpty',
                'sortNames' => array('mostRecent', 'mostPopularDiscussions', 'discussionsStarted', 'discussionsAddedTo'),
                'user' => $_GET['user']));
        $this->showListForContributorLinks = FALSE;
        $this->showContributorName = FALSE;
        $this->feedDescription = xg_text('SUBSCRIBE_TO_XS_DISCUSSIONS', $fullName);
        if ($_GET['user'] == XN_Profile::current()->screenName) {
            $this->noDiscussionsHtml = xg_html('YOU_HAVE_NOT_ADDED_DISCUSSIONS');
            if ($_GET['sort'] == 'discussionsStarted') {
                $this->noDiscussionsHtml = xg_html('YOU_HAVE_NOT_STARTED_DISCUSSIONS');
            }
            if ($_GET['sort'] == 'discussionsAddedTo') {
                $this->noDiscussionsHtml = xg_html('YOU_HAVE_NOT_ADDED_TO_DISCUSSIONS');
            }
        } else {
            $this->noDiscussionsHtml = xg_html('X_HAS_NOT_ADDED_DISCUSSIONS', xnhtmlentities($fullName));
            if ($_GET['sort'] == 'discussionsStarted') {
                $this->noDiscussionsHtml = xg_html('X_HAS_NOT_STARTED_DISCUSSIONS', xnhtmlentities($fullName));
            }
            if ($_GET['sort'] == 'discussionsAddedTo') {
                $this->noDiscussionsHtml = xg_html('X_HAS_NOT_ADDED_TO_DISCUSSIONS', xnhtmlentities($fullName));
            }
        }
    }

    /**
     * Prepares the fields for an action that lists topics, and sets the template to render. Sets $this->title, $this->pageSize,
     * $this->topicsAndComments, $this->totalCount, $this->feedUrl, and $this->pagePickerOptionsJson.
     *
     * Expected GET variables:
     *     page - page number (optional)
     *     sort - name of the current sort (optional), e.g., mostRecent
     *     feed - "yes" to output a feed (optional)
     *     tag - tag name to filter on (optional)
     *     q - search keywords (optional)
     *
     * @param $titleHtml string  HTML for the page heading
     * @param $templateForNoDiscussions string  Name of the template to use if there are no Topics or Comments
     * @param $sortNames array  Names of sorts to display in the combobox
     * @param $user string  (optional) Screen name of the person who started the topics
     * @param $category XN_Content|W_Content  (optional) Category to filter on
     * @param $useSearch boolean Whether to use a Search query (or not, use a Content query)
     */
    private function prepareListAction($args) {
        extract($args);
        XG_App::includeFileOnce('/lib/XG_FeedHelper.php');
        XG_App::includeFileOnce('/lib/XG_QueryHelper.php');
        XG_App::includeFileOnce('/lib/XG_SecurityHelper.php');
        XG_App::includeFileOnce('/lib/XG_PaginationHelper.php');
        $this->_widget->includeFileOnce('/lib/helpers/Forum_HtmlHelper.php');
        $this->_widget->includeFileOnce('/lib/helpers/Forum_CommentHelper.php');
        $this->_widget->includeFileOnce('/lib/helpers/Forum_Filter.php');
        $useSearch = isset($useSearch) ? $useSearch : false;
        $this->titleHtml = $titleHtml;
        $this->titleText = strip_tags(html_entity_decode($titleHtml, ENT_QUOTES, 'UTF-8'));
        $this->pageSize = 10;
        $begin = XG_PaginationHelper::computeStart($_GET['page'], $this->pageSize);
        $currentSortName = $_GET['sort'] ? $_GET['sort'] : $sortNames[0];

        /* If we're generating a feed, set the content type header up here before
         * we (possibly) get out early due to caching */
        if ($_GET['feed'] == 'yes') {
            header('Content-Type: application/atom+xml');
            $begin = 0;
        }

        if ($useSearch) {
            $query = XN_Query::create('Search');
        }
        else {
            // Don't cache tag queries as there are an infinite number of tags but a finite number of cache files [Jon Aquino 2007-02-09]
            $query = ($_GET['tag'] || $_GET['q'] || (! XG_Cache::cacheOrderN())) ? XN_Query::create('Content') : XG_Query::create('Content');
            if ($_GET['tag']) { $query->filter('tag->value', 'eic', $_GET['tag']); }
            if ($category) { Category::addCategoryFilter($query, $category); }
            /* If we want a feed and there's no tag query or no term query, then cache the feed */
            if ($_GET['feed'] && (mb_strlen($_GET['tag']) == 0) && (mb_strlen($_GET['q']) == 0)) {
                XG_FeedHelper::cacheFeed(array('id' => 'forum-topic-list-' . md5(XG_HttpHelper::currentUrl())));
            }
        }
        $query->begin($begin);
        $query->end($begin + $this->pageSize);
        if ($_GET['q']) { XG_QueryHelper::addSearchFilter($query, $_GET['q'], $useSearch); }
        // No need to call addExcludeFromPublicSearchFilter, as checkCurrentUserCanAccessGroup handles authorization [Jon Aquino 2007-04-25]
        if ($useSearch) {
            try {
                $searchResults = Forum_Filter::get('search')->execute($query, $user);
                $this->topicsAndComments = XG_QueryHelper::contentFromSearchResults($searchResults, false);
                $this->totalCount = $query->getTotalCount();
                /* If we're on the first page and all of the search results have been excluded because the
                 * matching content doesn't exist any more, then just pretend there are no results */
                if (($this->totalCount > 0) && (count($this->topicsAndComments) == 0) && ($begin ==0)) {
                    $this->totalCount = 0;
                }
            } catch (Exception $e) {
                if ($e->getMessage() == "Date range searching currently unsupported (BAZ-2459)") {
                    /* If we can't do a search query because we needed a date range, then fall back to content query */
                    $newArgs = $args;
                    $newArgs['useSearch'] = false;
                    return $this->prepareListAction($newArgs);
                } else {
                    /* Otherwise, just pretend there's no results and log that info */
                    error_log("Forum Topic search query ({$_GET['q']}) failed with: " . $e->getCode());
                    $this->topicsAndComments = array();
                    $this->totalCount = 0;
                }
            }
        }
        else {
            $this->topicsAndComments = Forum_Filter::get($currentSortName)->execute($query, $user);
            $this->totalCount = $query->getTotalCount();
        }
        $this->topics = Topic::topics($this->topicsAndComments);
        XG_Cache::profiles($this->topics, $this->topicsAndComments, $user, Topic::lastCommentContributorNames($this->topics));
        if ($_GET['feed'] == 'yes') {
            foreach ($this->topicsAndComments as $topicOrComment) {
                if ($topicOrComment->type == 'Comment') { $topicOrComment->title = $this->topics[$topicOrComment->my->attachedTo]->title; }
            }
            XG_FeedHelper::outputFeed($this->topicsAndComments, $this->titleText, $user ? XG_Cache::profiles($user) : NULL);
            exit;
        }
        $this->feedUrl = XG_HttpHelper::addParameters(XG_HttpHelper::currentUrl(), array('feed' => 'yes', 'xn_auth' => 'no'));
        if ($currentSortName == 'mostRecentlyUpdatedDiscussions' && $category == null) {
            $this->feedUrlReplies = XG_HttpHelper::addParameters(XG_HttpHelper::currentUrl(), array('sort' => 'mostRecent', 'feed' => 'yes', 'xn_auth' => 'no'));
        } else {
            $this->feedUrl = XG_HttpHelper::addParameters($this->feedUrl, array('sort' => 'mostRecent'));
        }
        $this->pagePickerOptionsJson = self::pagePickerOptionsJson($sortNames, $currentSortName, $user);
        $this->showFeedLink = ! XG_App::appIsPrivate() && ! XG_GroupHelper::groupIsPrivate();
        if (!$this->totalCount || $_GET['test_empty']) {
            $this->render($templateForNoDiscussions);
        } else {
            $this->render('list');
        }
    }

    /**
     * Returns JSON for the given sorts, for initializing the combobox.
     *
     * @param $sortNames array  Names of sorts to display in the combobox
     * @param $currentSortName string  Name of the current sort
     * @param $username string  Username that will be filtered on (optional)
     * @return string  JSON to pass to the PagePicker
     */
    private function pagePickerOptionsJson($sortNames, $currentSortName, $username) {
        $filterMetadata = array();
        foreach ($sortNames as $sortName) {
            $filterMetadata[] = array(
                    'displayText' => Forum_Filter::get($sortName)->getDisplayText($username),
                    'url' => XG_HttpHelper::addParameter(XG_HttpHelper::currentUrl(), 'sort', $sortName),
                    'selected' => $sortName == $currentSortName);
        }
        $json = new NF_JSON();
        return $json->encode($filterMetadata);
    }

    /**
     * Returns JSON for the given categories, for initializing the combobox.
     *
     * @param $categories array  The Category objects
     * @param $currentCategory XN_Content|W_Content  Category for the current Topic
     * @return string  JSON to pass to the CategoryPicker
     */
    private function categoryPickerOptionsJson($categories, $currentCategory) {
        $filterMetadata = array();
        foreach ($categories as $category) {
            $filterMetadata[] = array('displayText' => $category->title, 'id' => $category->id);
        }
        $json = new NF_JSON();
        return $json->encode($filterMetadata);
    }

    /**
     * Updates the topic's category
     *
     * Expected GET variables:
     *     id - ID of the Topic to edit
     *
     * Expected POST variables:
     *     categoryId - ID of the new Category
     */
    public function action_setCategory() {
        $this->_widget->includeFileOnce('/lib/helpers/Forum_SecurityHelper.php');
        $this->_widget->includeFileOnce('/lib/helpers/Forum_HtmlHelper.php');
        $topic = XG_GroupHelper::checkCurrentUserCanAccess(W_Content::load($_GET['id']));
        if ($topic->type != 'Topic') { throw new Exception('Not a Topic'); }
        if (! Forum_SecurityHelper::currentUserCanSetCategory($topic)) { throw new Exception('Not allowed'); }
        if ($_SERVER['REQUEST_METHOD'] != 'POST') { throw new Exception('Not a POST'); }
        $category = Category::find($_POST['categoryId']);
        if (! $category) { throw new Exception('Category not found'); }
        Category::setCategoryId($topic, $category->id);
        $topic->save();
        $this->redirectTo('show', 'topic', array('id' => $topic->id));
    }

    /**
     * Ends commenting on the discussion thread.
     *
     * Expected GET variables:
     *     id - ID of the Topic for which to close comments
     *     target - URL to redirect to afterwards
     */
    public function action_closeComments() {
        $this->_widget->includeFileOnce('/lib/helpers/Forum_SecurityHelper.php');
        $topic = XG_GroupHelper::checkCurrentUserCanAccess(W_Content::load($_GET['id']));
        if ($topic->type != 'Topic') { throw new Exception('Not a Topic (2047548834)'); }
        if (! Forum_SecurityHelper::currentUserCanCloseComments($topic)) { throw new Exception('Not allowed (2123475459)'); }
        if ($_SERVER['REQUEST_METHOD'] != 'POST') { throw new Exception('Not a POST (630114808)'); }
        $topic->my->commentsClosed = 'Y';
        $topic->save();
        $this->redirectTo($_GET['target']);
    }

    /**
     * Re-opens commenting on the discussion thread.
     *
     * Expected GET variables:
     *     id - ID of the Topic for which to open comments
     *     target - URL to redirect to afterwards
     */
    public function action_openComments() {
        $this->_widget->includeFileOnce('/lib/helpers/Forum_SecurityHelper.php');
        $topic = XG_GroupHelper::checkCurrentUserCanAccess(W_Content::load($_GET['id']));
        if ($topic->type != 'Topic') { throw new Exception('Not a Topic (1576619427)'); }
        if (! Forum_SecurityHelper::currentUserCanOpenComments($topic)) { throw new Exception('Not allowed (1668656309)'); }
        if ($_SERVER['REQUEST_METHOD'] != 'POST') { throw new Exception('Not a POST (720688204)'); }
        $topic->my->commentsClosed = 'N';
        $topic->save();
        $this->redirectTo($_GET['target']);
    }

}
