<?php

/**
 * Dispatches requests pertaining to categories.
 */
class Forum_CategoryController extends XG_GroupEnabledController {

    /**
     * Displays the form for editing categories
     */
    public function action_edit() {
        XG_SecurityHelper::redirectIfNotMember();
        $this->_widget->includeFileOnce('/lib/helpers/Forum_SecurityHelper.php');
        $this->_widget->includeFileOnce('/lib/helpers/Forum_Filter.php');
        if (! Forum_SecurityHelper::currentUserCanEditCategories()) { throw new Exception('Not allowed (689672525)'); }
        $this->topicCounts = array();
        $this->categories = Category::findAll();
        if (count($this->categories) == 0) {
            $this->categories[] = $sample = Category::create(xg_text('SAMPLE_TITLE'));
            $sample->my->membersCanAddTopics = 'Y';
            $sample->my->membersCanReply = 'Y';
            $this->topicCounts[] = 0;
            Forum_Filter::get('mostRecentDiscussions')->execute($query = XN_Query::create('Content')->end(1));
            if ($query->getTotalCount()) {
                $this->categories[] = $uncategorized = Category::create(xg_text('UNCATEGORIZED'), xg_text('EXISTING_DISCUSSIONS'));
                $this->topicCounts[] = $query->getTotalCount();
                $uncategorized->my->membersCanAddTopics = 'Y';
                $uncategorized->my->membersCanReply = 'Y';
                $uncategorized->my->alternativeIds = 'null';
                // Yes, this should be the string 'null' - see Category#alternativeIds [Jon Aquino 2007-03-30]
            }
        } else {
            foreach ($this->categories as $category) {
                $query = XN_Query::create('Content')->end(1);
                Category::addCategoryFilter($query, $category);
                Forum_Filter::get('mostRecentDiscussions')->execute($query);
                $this->topicCounts[] = $query->getTotalCount();
            }
        }
    }

    /**
     * Processes the form for editing categories.
     */
    public function action_update() {
        XG_SecurityHelper::redirectIfNotMember();
        $this->_widget->includeFileOnce('/lib/helpers/Forum_SecurityHelper.php');
        XG_App::includeFileOnce('/lib/XG_SecurityHelper.php');
        if (! Forum_SecurityHelper::currentUserCanEditCategories()) { throw new Exception('Not allowed (1496321638)'); }
        if (! $_POST['data']) { throw new Exception('Data empty'); }
        $json = new NF_JSON(SERVICES_JSON_LOOSE_TYPE);
        $categories = Category::buildCategories($json->decode($_POST['data']));
        if (count($categories) == 0) {
            // If all categories are deleted, we will be reverting to the non-category homepage layout.
            // But we don't want to show any discussions created before this point, as they
            // are supposed to be deleted [Jon Aquino 2007-03-30]
            // allCategoriesDeletedOn would have been unnecessary if we had implemented BAZ-2459 [Jon Aquino 2007-09-01]
            $this->_widget->privateConfig['allCategoriesDeletedOn'] = date('c', time());
            $this->_widget->saveConfig();
        }
        $this->redirectTo('index', 'index');
    }

    /**
     * Displays a list of recent discussions in specific categories (paginated).
     *
     * Optional $_GET variable: 'page' for which page of discussions to view.  Defaults to 1.
     */
    public function action_list() {
        $this->_widget->includeFileOnce('/lib/helpers/Forum_SecurityHelper.php');
        $this->_widget->includeFileOnce('/lib/helpers/Forum_Filter.php');
        XG_App::includeFileOnce('/lib/XG_PaginationHelper.php');
        $CATEGORIES_PER_PAGE = 5;
        $TOPICS_PER_CATEGORY = 5;
        $pageNum = isset($_GET['page']) ? (integer) $_GET['page'] : 1;
        //TODO more efficient to change findAll to call a new find function that can partition and then call that new function here.
        $this->numPerPage = $CATEGORIES_PER_PAGE;
        $offset = ($pageNum - 1) * $CATEGORIES_PER_PAGE;
        $allCategories = Category::findAll();
        $this->totalCategories = count($allCategories);
        $this->categories = array_slice($allCategories, $offset, $CATEGORIES_PER_PAGE);
        $this->categoryIdToRecentTopics = array();
        $fetchTopicsCount = 5;
        $i = 0;
        $topics = array();
        foreach ($this->categories as $category) {
            $i++;
            if ($i <= $fetchTopicsCount) {
                $this->categoryIdToRecentTopics[$category->id] = Category::recentTopics($category, $TOPICS_PER_CATEGORY);
                $topics = array_merge($topics, $this->categoryIdToRecentTopics[$category->id]);
	        } else {
	            $this->categoryIdToRecentTopics[$category->id] = array(-1);
	        }
        }
        $this->showFeedLink = ! XG_App::appIsPrivate() && ! XG_GroupHelper::groupIsPrivate();
        $this->showFollowLink = Forum_SecurityHelper::currentUserCanFollowNewTopics();
        // Pre-load profiles and User objects [Jon Aquino 2007-10-07]
        XG_Cache::profiles($topics, Topic::lastCommentContributorNames($topics));
    }
    
    /**
     * Paginated list of all categories.
     *
     * Optional $_GET variable: 'page' for which page of categories to view.  Defaults to 1.
     */
    public function action_listCategories() {
        XG_App::includeFileOnce('/lib/XG_PaginationHelper.php');
        $categories = Category::findAll();
        $CATEGORIES_PER_PAGE = 50;
        $pageNum = isset($_GET['page']) ? (integer) $_GET['page'] : 1;
        $this->numPerPage = $CATEGORIES_PER_PAGE;
        $offset = ($pageNum - 1) * $CATEGORIES_PER_PAGE;
        $allCategories = Category::findAll();
        $this->totalCategories = count($allCategories);
        $this->categories = array_slice($allCategories, $offset, $CATEGORIES_PER_PAGE);
    }
}
