<?php

/**
 * Useful functions for working with XN_Jobs and XN_Tasks.
 */
class  XG_JobHelper {

    /**
     * Initiates a sequenced job. A sequenced job is an XN_Job that
     * will create a new XN_Job if it cannot complete its work. This results in a
     * sequence (or "chain") of jobs.
     *
     * To create a sequenced job, ensure that your widget has a controller named
     * SequencedjobController, which extends XG_SequencedjobController.
     * Add an action that does the work; it should call setContinueJob() to indicate
     * whether to create a new job or to stop.
     *
     * To run the sequenced job, call XG_JobHelper::start(widget, action, args).
     * The args will be passed to your action as instance variables. If you modify
     * or add new instance variables, the new values will be passed to the next job.
     *
     * @param $widgetName string  the name of the widget, e.g., profiles
     * @param $actionName string  the name of the action, e.g., buildBlogArchive
     * @param $args array  names and values to pass to the action as instance variables
     */
    public static function start($widgetName, $actionName, $args = array()) {
        self::run(W_Cache::getWidget($widgetName)->buildUrl('sequencedjob', $actionName), $args);
    }

    /**
     * Determines if an asynchronous job should be allowed to create another job.
     * Should be used before scheduling a potentially long loop of jobs so that they can be manually stopped in emergencies.
     *
     * @return  boolean true if chaining is allowed at this time.
     */
    public static function allowChaining() {
        if (file_exists('xn-app://socialnetworkmain/xn/XG_JobHelper.stopChaining.txt')) { return false; } // global check
        if (file_exists(NF_APP_BASE . '/xn_private/XG_JobHelper.stopChaining.txt')) { return false; } // local check
        return true;
    }

    /**
     * Creates and executes an asynchronous job.
     *
     * Example:
     *     XG_JobHelper::run('http://networkname.ning.com/foo', array('name' => 'Paul', 'city' => 'Victoria'));
     *
     * If $type is not set, it is chosen automatically as follows:
     *     - If $content is an array, the $type will be application/x-www-form-urlencoded.
     *     - If $content is valid XML, the $type will be application/xml.
     *     - Otherwise, the $type will be application/octet-stream.
     *
     * @param $executionCallback string  the URL to post to
     * @param $content array|string  the data for the POST body
     * @param $type string  the MIME type of the data, or null to choose one automatically.
     */
    private static function run($executionCallback, $content = array(), $type = null) {
        $job = XN_Job::create();
        $job->addTask(XN_Task::create($executionCallback, $content, $type));
        $result = $job->save();
        W_Cache::getWidget('main')->includeFileOnce('/lib/helpers/Index_InvitationHelper.php');
        if (Index_InvitationHelper::isErrorArray($result)) { throw new Exception(Index_InvitationHelper::errorMessage(key($result))); }
    }

}
