<?php

/**
 * Useful functions for working with sign-in and sign-up.
 */
class XG_AuthorizationHelper {

    /**
     * Returns the URL for the sign-in page.
     *
     * @param $target string  the URL to go to after sign-in, or null for the current page
     * @param $groupToJoin string|XN_Content|W_Content  (optional) the Group object (or its URL) to make the user a member of
     * @param string emailAddress (optional) initial value for the Email Address field, or null to use that of the current user (if any)
     * @return string  the URL
     */
    public static function signInUrl($target = null, $groupToJoin = null, $emailAddress = null) {
        if ($groupToJoin instanceof XN_Content || $groupToJoin instanceof W_Content) { $groupToJoin = $groupToJoin->my->url; }
        if (! $target) { $target = XG_HttpHelper::currentUrl(); }
        return W_Cache::getWidget('main')->buildUrl('authorization', 'signIn', array('target' => self::addSourceParameter($target), 'groupToJoin' => $groupToJoin, 'emailAddress' => $emailAddress));
    }

    /**
     * Returns the URL for the sign-up page.
     *
     * @param $target string  the URL to go to after sign-up, or null for the current page
     * @param $groupToJoin string|XN_Content|W_Content  (optional) the Group object (or its URL) to make the user a member of
     * @param string emailAddress (optional) initial value for the Email Address field, or null to use that of the current user (if any)
     * @return string  the URL
     */
    public static function signUpUrl($target = null, $groupToJoin = null, $emailAddress = null) {
        if ($groupToJoin instanceof XN_Content || $groupToJoin instanceof W_Content) { $groupToJoin = $groupToJoin->my->url; }
        if (! $target) { $target = XG_HttpHelper::currentUrl(); }
        return W_Cache::getWidget('main')->buildUrl('authorization', 'signUp', array('target' => self::addSourceParameter($target), 'groupToJoin' => $groupToJoin, 'emailAddress' => $emailAddress));
    }

    /**
     * Returns the URL for the sign-up-Ning-user page.
     *
     * @param $target string  the URL to go to after sign-up, or null for the current page
     * @param $groupToJoin string|XN_Content|W_Content  (optional) the Group object (or its URL) to make the user a member of
     * @param string emailAddress (optional) initial value for the Email Address field, or null to use that of the current user (if any)
     * @return string  the URL
     */
    public static function signUpNingUserUrl($target = null, $groupToJoin = null, $emailAddress = null) {
        if ($groupToJoin instanceof XN_Content || $groupToJoin instanceof W_Content) { $groupToJoin = $groupToJoin->my->url; }
        if (! $target) { $target = XG_HttpHelper::currentUrl(); }
        return W_Cache::getWidget('main')->buildUrl('authorization', 'signUpNingUser', array('target' => self::addSourceParameter($target), 'groupToJoin' => $groupToJoin, 'emailAddress' => $emailAddress));
    }

    /**
     * Returns the URL for signing out.
     *
     * @param $target string  the URL to go to after sign-out, or null for the homepage
     * @return string  the URL
     */
    public static function signOutUrl($target = null) {
        if (! $target) { $target = xg_absolute_url('/'); }
        return W_Cache::getWidget('main')->buildUrl('authorization', 'signOut', array('target' => $target, XG_SecurityHelper::CSRF_TOKEN_NAME => XG_SecurityHelper::getCsrfToken()));
    }

    /**
     * Returns whether the email-password combination is valid.
     *
     * @param $screenNameOrEmail string  a Ning username or email address
     * @param $password string  a password to check
     * @return mixed  true if valid; otherwise an array of error messages keyed by numeric error codes
     */
    public static function verifyPassword($screenNameOrEmail, $password) {
        return XN_Profile::signIn($screenNameOrEmail, $password, array('set-cookies' => false));
    }

    /**
     * Redirects to the PIN-verification form if necessary.
     */
    public static function redirectIfPinRequired() {
      if (XN_Profile::current()->loginIsVerified() != XN_Profile::IS_VERIFIED && XG_SecurityHelper::userIsAdmin() && !XG_App::pinOptional(XG_App::getRequestedRoute())) {
          $url = XN_Profile::verificationUrl(XG_HttpHelper::currentUrl());
          header('Location: '.$url, true, 302);
          exit;
        }
    }

    /**
     * Set's the URL's "source" URL parameter to the current page's URL.
     * The source is where the Membership Pending page's Continue link will go.
     *
     * @param $url string  the original URL, or null
     * @return string  the URL with the source parameter added (if missing), or null if $url is null
     */
    protected static function addSourceParameter($url, $currentUrl = null) {
        // TODO: Eliminate the source parameter; use a cookie instead (BAZ-5020) [Jon Aquino 2008-01-09]
        if (! $url || XG_HttpHelper::hasParameter($url, 'source')) { return $url; }
        if (! $currentUrl) { $currentUrl = XG_HttpHelper::currentUrl(); }
        if (XG_HttpHelper::removeParameter($currentUrl, XG_App::SIGN_IN_IF_SIGNED_OUT) == $url) { return $url; }
        if (self::isAuthorizationUrl($currentUrl)) { return $url; }
        return XG_HttpHelper::addParameter($url, 'source', $currentUrl);
    }

    /**
     * Returns whether the URL is for an action in the AuthorizationController.
     *
     * @param $url string  the URL to examine
     * @return boolean  whether the URL has the pattern /main/authorization/*
     */
    private static function isAuthorizationUrl($url) {
        return mb_strpos($url, 'main/authorization') !== false;
    }

}
