<?php
/**
 * Dispatches requests pertaining to replies to discussion pages.
 */
XG_App::includeFileOnce('/lib/XG_CommentHelper.php'); 
 
class Page_CommentController extends W_Controller {

    protected function _before() {
        $this->_widget->includeFileOnce('/lib/helpers/Page_SecurityHelper.php');
    }

    /**
     * Processes the form for a new comment on a page.
     *
     * Expected GET variables:
     *     page - ID of the Page object
     */
    public function action_create() {
        XG_SecurityHelper::redirectIfNotMember();
        if (! $_POST['comment']) { throw new Exception('Empty comment (1175663871)'); }
        $page = W_Content::load($_GET['page']);
        if ($page->type != 'Page') { throw new Exception('Not a Page'); }
        $comment = Comment::createAndAttachTo($page, xg_scrub(mb_substr($_POST['comment'], 0, 4000)));
        $comment->save();
        if ($_GET['xn_out'] != 'json') { return $this->redirectTo('show', 'page', array('id' => $page->id)); }
        XG_App::includeFileOnce('/lib/XG_CommentHelper.php');
        ob_start();
        XG_CommentHelper::outputComment(array(
                'comment' => $comment,
                'canDelete' => Page_SecurityHelper::currentUserCanDeleteComment($comment),
                'deleteEndpoint' => $this->_buildUrl('comment','delete', array('xn_out' => 'json')),
                'canApprove' => false,
                'approveEndpoint' => null));
        $this->html = trim(ob_get_contents());
        $this->approved = true;
        $this->userIsNowFollowing = false;
        ob_end_clean();
    }

    /**
     * Returns JSON containing HTML for a Comment
     *
     * Expected GET variables:
     *     id - ID of the Comment to display
     *     xn_out - Set this to "json"
     */
    public function action_show() {
        $this->_widget->includeFileOnce('/lib/helpers/Page_FileHelper.php');
        $this->_widget->includeFileOnce('/lib/helpers/Page_HtmlHelper.php');
        $this->_widget->includeFileOnce('/lib/helpers/Page_SecurityHelper.php');
        $this->_widget->includeFileOnce('/lib/helpers/Page_CommentHelper.php');
        $comment = W_Content::load($_GET['id']);
        if ($comment->type != 'Comment') { throw new Exception('Not a Comment'); }
        $page = W_Content::load($comment->my->attachedTo);
        if ($page->type != 'Page') { throw new Exception('Not a Page'); }
        ob_start();
        $this->renderPartial('fragment_comment', 'page', array('page' => $page, 'comment' => $comment));
        $this->html = trim(ob_get_contents());
        ob_end_clean();
    }

    /**
     * Processes the form for editing a comment
     *
     * Expected GET variables:
     *     id - ID of the Comment to edit
     */
    public function action_update() {
        $json = new NF_JSON();
        header('Content-Type: text/plain');
        try {
            XG_App::includeFileOnce('/lib/XG_HttpHelper.php');
            $this->_widget->includeFileOnce('/lib/helpers/Page_SecurityHelper.php');
            $this->_widget->includeFileOnce('/lib/helpers/Page_HtmlHelper.php');
            $this->_widget->includeFileOnce('/lib/helpers/Page_CommentHelper.php');
            if ($_SERVER['REQUEST_METHOD'] != 'POST') { throw new Exception('Not a POST'); }
            XG_HttpHelper::trimGetAndPostValues();
            $comment = W_Content::load($_GET['id']);
            if ($comment->type != 'Comment') { throw new Exception('Not a Comment'); }
            if (! Page_SecurityHelper::currentUserCanEditComment($comment)) { throw new Exception('Not allowed'); }
            $description = Page_CommentHelper::cleanDescription($_POST['value']);
            if ($description != xg_text('NO_DESCRIPTION')) {
                $comment->description = $description;
                $comment->save();
            }
            echo '(' . $json->encode(array('html' => xg_resize_embeds($comment->description, 646))) . ')';
        } catch (Exception $e) {
            echo '(' . $json->encode(array('friendlyErrorMessage' => xg_text('HMM_SOMETHING_WENT_AWRY'))) . ')';
        }
    }

    /**
     * Deletes the comment, then redirects to the discussion page.
     *
     * Expected GET variables:
     *     id - ID of the Comment object to delete
     */
    public function action_delete() {
        $this->_widget->includeFileOnce('/lib/helpers/Page_SecurityHelper.php');
        $this->_widget->includeFileOnce('/lib/helpers/Page_FileHelper.php');
        $this->_widget->includeFileOnce('/lib/helpers/Page_UserHelper.php');
        if ($_SERVER['REQUEST_METHOD'] != 'POST') { throw new Exception('Not a POST'); }
        $comment = XN_Content::load($_POST['id']);
        if ($comment->type != 'Comment') { throw new Exception('Not a Comment'); }
        if (! Page_SecurityHelper::currentUserCanDeleteComment($comment)) { throw new Exception('Not allowed'); }
        $pageId = $comment->my->attachedTo;
        Comment::remove($comment);
        Page_UserHelper::updateActivityCount(User::load($this->_user))->save();
        $this->success = true;
    }

}
