<?php

class Index_SharingController extends W_Controller {

    /**
     * Runs code before each action.
     */
    protected function _before() {
        W_Cache::getWidget('main')->includeFileOnce('/lib/helpers/Index_InvitationFormHelper.php');
        $this->_widget->includeFileOnce('/lib/helpers/Index_SharingHelper.php');
        XG_HttpHelper::trimGetAndPostValues();
        $_GET['id'] = urldecode($_GET['id']);  // Is this needed?  [Jon Aquino 2007-10-26]
    }

    /**
     * Display the 'share this' page for the specified object
     *
     * Expected GET params:
     *     id - content object ID
     *     sent - whether messages were just sent
     *     noAddressesFound - whether the address import found 0 addresses
     *
     * @param $formToOpen string  (optional) which form to open: enterEmailAddresses, inviteFriends, webAddressBook, or emailApplication
     * @param $errors array  (optional) HTML error messages, optionally with keys field name
     */
    public function action_share($formToOpen = 'enterEmailAddresses', $errors = array()) {
        $this->_widget->includeFileOnce('/lib/helpers/Index_MessageHelper.php');
        XG_App::includeFileOnce('/lib/XG_ContactHelper.php');
        if (! $this->_user->isLoggedIn()) { return $this->forwardTo('shareSignedOut'); }
        $this->itemInfo = self::getItemInfo($_GET['id']);
        self::verifyUserCanShare($this->itemInfo['object']);
        $this->pageTitle = Index_SharingHelper::pageTitle($this->itemInfo);
        $this->showInvitationsSentMessage = $_GET['sent'];
        $this->showNoAddressesFoundMessage = $_GET['noAddressesFound'];
        $numFriends = Index_MessageHelper::numberOfFriendsAcrossNing($this->_user->screenName);
        $this->invitationArgs = array(
                'formToOpen' => $formToOpen,
                'errors' => $errors,
                'createUrl' => $this->_buildUrl('sharing', 'create', array('id' => $_GET['id'])),
                'enterEmailAddressesButtonText' => xg_text('SEND_MESSAGE'),
                'inviteFriendsTitle' => xg_text('SHARE_WITH_FRIENDS'),
                'inviteFriendsDescription' => xg_text('SHARE_THIS_WITH_FRIENDS'),
                'friendDataUrl' => $this->_buildUrl('sharing', 'friendData', array('xn_out' => 'json')),
                'numFriends' => $numFriends,
				'messageParts' => $this->itemInfo['message_parts'],
                'showInviteFriendsForm' => $numFriends > 0);
    }

    /**
     * Outputs JSON for "friends" (each with screenName, fullName, thumbnailUrl,
     * and optional reasonToDisable) and "paginationHtml".
     *
     * Expected GET variables
     *     xn_out - "json";
     *     page - 1, 2, 3, ...
     */
    public function action_friendData() {
        $this->_widget->includeFileOnce('/lib/helpers/Index_MessageHelper.php');
        $friendData = Index_MessageHelper::dataForFriendsAcrossNing($_GET['page']);
        $this->friends = $friendData['friends'];
        $this->paginationHtml = $friendData['paginationHtml'];
    }

    /**
     * Processes the form for sending invitations.
     *
     * Expected GET params:
     *     id - content object ID
     *
     * Expected POST variables:
     *
     *     form - "enterEmailAddresses"
     *     emailAddresses - email addresses separated by commas, semicolons, and whitespace
     *     message - optional message for the invitation
     *
     * or
     *
     *     form - "inviteFriends"
     *     allFriends - whether to send the message to all friends of the current user
     *     screenNames - JSON array of screen names of friends (if allFriends is false)
     *     inviteFriendsMessage - optional message for the invitation
     *
     * or
     *
     *     form - "webAddressBook"
     *     emailLocalPart - the part of the email address before the "@"
     *     emailDomain - the part of the email address after the "@"
     *     password - the password for the email address
     *
     * or
     *
     *     form - "emailApplication"
     *     file - a file containing CSV or VCF data
     */
    public function action_create() {
        if (! $this->_user->isLoggedIn()) { return $this->forwardTo('shareSignedOut'); }
        $this->itemInfo = self::getItemInfo($_GET['id']);
        self::verifyUserCanShare($this->itemInfo['object']);
        if ($_SERVER['REQUEST_METHOD'] != 'POST') { return $this->redirectTo('share', 'sharing', array('id' => $_GET['id'])); }
        switch ($_POST['form']) {

            case 'enterEmailAddresses':
                $result = Index_InvitationFormHelper::processEnterEmailAddressesForm();
                if ($result['errorHtml']) { return $this->forwardTo('share', 'sharing', array('enterEmailAddresses', array($result['errorHtml']))); }
                Index_InvitationFormHelper::send(array(
                        'inviteOrShare' => 'share',
                        'groupId' => Index_SharingHelper::groupId($this->itemInfo),
                        'contactList' => $result['contactList'],
                        'message' => $_POST['message'],
                        'contentId' => $_GET['id']));
                $this->redirectTo('share', 'sharing', array('id' => $_GET['id'], 'sent' => 1));
                break;

            case 'inviteFriends':
                $result = Index_InvitationFormHelper::processInviteFriendsForm();
                if ($result['errorHtml']) { return $this->forwardTo('share', 'sharing', array('inviteFriends', array($result['errorHtml']))); }
                Index_InvitationFormHelper::send(array(
                        'inviteOrShare' => 'share',
                        'groupId' => Index_SharingHelper::groupId($this->itemInfo),
                        'allFriends' => $result['allFriends'],
                        'contactList' => $result['contactList'],
                        'message' => $_POST['inviteFriendsMessage'],
                        'contentId' => $_GET['id']));
                $this->redirectTo('share', 'sharing', array('id' => $_GET['id'], 'sent' => 1));
                break;

            case 'webAddressBook':
                $result = Index_InvitationFormHelper::processWebAddressBookForm();
                if ($result['errorHtml']) { return $this->forwardTo('share', 'sharing', array('webAddressBook', array($result['errorHtml']))); }
                $this->redirectTo($result['target']);
                break;

            case 'emailApplication':
                $result = Index_InvitationFormHelper::processEmailApplicationForm();
                if ($result['errorHtml']) { return $this->forwardTo('share', 'sharing', array('emailApplication', array($result['errorHtml']))); }
                $this->redirectTo($result['target']);
                break;
        }
    }

    /**
     * Displays an AJAX-based form for editing the list of recipients for the invitation.
     *
     * Expected GET params:
     *     id - content object ID
     *
     * Expected GET variables:
     *     contactListId - content ID of a ContactList object
     */
    public function action_editContactList() {
        if (! $this->_user->isLoggedIn()) { return $this->forwardTo('shareSignedOut'); }
        if (! unserialize(ContactList::load($_GET['contactListId'])->my->contacts)) { return $this->redirectTo('share', 'sharing', array('noAddressesFound' => 1, 'id' => $_GET['id'])); }
        $this->itemInfo = self::getItemInfo($_GET['id']);
        self::verifyUserCanShare($this->itemInfo['object']);
        $this->pageTitle = Index_SharingHelper::pageTitle($this->itemInfo);
        $this->invitationArgs = array(
                'contactListId' => $_GET['contactListId'],
                'createWithContactListUrl' => $this->_buildUrl('sharing', 'createWithContactList', array('contactListId' => $_GET['contactListId'], 'id' => $_GET['id'])),
                'cancelUrl' => Index_SharingHelper::url($this->itemInfo),
                'inviteOrShare' => 'share',
                'searchLabelText' => xg_text('SEARCH_FRIENDS'),
                'messageParts' => $this->itemInfo['message_parts'],
                'submitButtonText' => xg_text('SEND'));
    }

    /**
     * Processes the Contact List form.
     *
     * Expected GET variables:
     *     contactListId - content ID of a ContactList object
     *     id - content object ID
     *
     * Expected POST variables:
     *     contactListJson - a JSON array of contacts, each being an array with keys "name" and "emailAddress"
     *     message - optional message for the invitation
     */
    public function action_createWithContactList() {
        if (! $this->_user->isLoggedIn()) { return $this->forwardTo('shareSignedOut'); }
        $this->itemInfo = self::getItemInfo($_GET['id']);
        self::verifyUserCanShare($this->itemInfo['object']);
        if ($_SERVER['REQUEST_METHOD'] != 'POST') { return $this->redirectTo('share', 'sharing', array('id' => $_GET['id'])); }
        Index_InvitationFormHelper::processContactListForm('share', Index_SharingHelper::groupId($this->itemInfo));
        $this->redirectTo('share', 'sharing', array('id' => $_GET['id'], 'sent' => 1));
    }

    public function action_shareSignedOut() {
        XG_App::includeFileOnce('/lib/XG_Message.php');
        if ($this->_user->isLoggedIn()) { return $this->forwardTo('share'); }
        $this->itemInfo = (self::getItemInfo($_GET['id']));
        self::verifyCanDisplayToLoggedOut($this->itemInfo['object']);
        $this->pageTitle = Index_SharingHelper::pageTitle($this->itemInfo);
        $message = Index_SharingHelper::createMessage($this->itemInfo, $message);
        $body = $message->build(null, null, false);
        $this->mailToLink = xg_mailto_url(Index_SharingHelper::subject($this->itemInfo), $body);
    }

    private function verifyUserCanShare($obj) {
        if (! Index_SharingHelper::userCanShare($obj)) {
            header('Location: ' . xg_absolute_url('/'));
            exit;
        }
    }

    /**
     * can this item be displayed on the logged out share page?
     */
    private function verifyCanDisplayToLoggedOut($obj) {
        if (! Index_SharingHelper::canDisplayToLoggedOut($obj)) {
            header('Location: ' . xg_absolute_url('/'));
            exit;
        }
    }

    /**
     *  Get relevant sharing information for the object with the specified ID.
     *
     *  IMPORTANT:  This function will show the user a 404 page if the object is not found.
     *
     * @param $id string
     */
    private function getItemInfo($id) {
        $itemInfo = Index_SharingHelper::getItemInfo($id);
        if (! $itemInfo) {
            W_Cache::getWidget('main')->dispatch('error','404');
            exit;
        }
        return $itemInfo;
    }
}
