<?php

/**
 * Dispatches requests pertaining to composing messages to friends.
 */
class Index_MessageController extends W_Controller {

    /**
     * Displays a form for sending messages to friends.
     *
     * Expected GET variables:
     *     sent - whether messages were just sent
     */
    public function action_new() {
        $this->_widget->includeFileOnce('/lib/helpers/Index_MessageHelper.php');
        XG_SecurityHelper::redirectIfNotMember();
        $this->showMessagesSentMessage = $_GET['sent'];
        $this->friendListArgs = array(
                'friendDataUrl' => $this->_buildUrl('message', 'friendData', array('xn_out' => 'json')),
                'numFriends' => Index_MessageHelper::numberOfFriendsOnNetwork($this->_user->screenName),
                'allFriendsButtonText' => xg_text('ALL_FRIENDS_ON_APPNAME', XN_Application::load()->name));
    }

    /**
     * Processes the form for sending a message to friends.
     *
     * Expected POST variables:
     *     allFriends - whether to send the message to all friends of the current user
     *     screenNames - JSON array of screen names of friends (if allFriends is false)
     *     subject - the subject line
     *     message - the message text
     */
    public function action_create() {
        $this->_widget->includeFileOnce('/lib/helpers/Index_MessageHelper.php');
        $this->_widget->includeFileOnce('/lib/helpers/Index_InvitationHelper.php');
        XG_App::includeFileOnce('/lib/XG_ContactHelper.php');
        XG_SecurityHelper::redirectIfNotMember();
        XG_HttpHelper::trimGetAndPostValues();
        $json = new NF_JSON(SERVICES_JSON_LOOSE_TYPE);
        if ($_SERVER['REQUEST_METHOD'] != 'POST') { return $this->redirectTo('new', 'message'); }
        $subject = mb_substr($_POST['subject'], 0, Index_MessageHelper::MAX_SUBJECT_LENGTH);
        $message = mb_substr($_POST['message'], 0, Index_MessageHelper::MAX_MESSAGE_LENGTH);
        if ($subject || $message) {
            if ($_POST['allFriends']) {
                $job = Index_MessageHelper::createJobForAllFriends($subject, $message, Index_MessageHelper::numberOfFriendsOnNetwork($this->_user->screenName));
            } else {
                $job = Index_MessageHelper::createJobForScreenNames($subject, $message, $json->decode($_POST['screenNames']));
            }
            $result = $job->save();
            if (Index_InvitationHelper::isErrorArray($result)) { throw new Exception(Index_InvitationHelper::errorMessage(key($result))); }
        }
        $this->redirectTo('new', 'message', array('sent' => 1));
    }

    /**
     * Displays a list of friends for the current user.
     *
     * @param $args array:
     *     allFriendsButtonText - text for the All Friends radio button
     *     friendDataUrl - endpoint for retrieving friend info
     *     numFriends - total number of friends
     */
    public function action_friendList($args) {
        foreach ($args as $key => $value) { $this->{$key} = $value; }
        $this->form = new XNC_Form(array('allFriends' => '1'));
    }

    /**
     * Outputs JSON for "friends" (each with screenName, fullName, thumbnailUrl,
     * and optional reasonToDisable) and "paginationHtml".
     *
     * Expected GET variables
     *     xn_out - "json";
     *     page - 1, 2, 3, ...
     */
    public function action_friendData() {
        $this->_widget->includeFileOnce('/lib/helpers/Index_MessageHelper.php');
        $friendData = Index_MessageHelper::dataForFriendsOnNetwork($_GET['page']);
        $this->friends = $friendData['friends'];
        $this->paginationHtml = $friendData['paginationHtml'];
    }

    /**
     * Sends messages to the specified friends.
     *
     * Expected POST variables:
     *
     *     screenNames - array of screen names of friends
     *     begin - begin index for a friend query (alternative to screenNames)
     *     end - end index for a friend query (alternative to screenNames)
     *     subject - the subject line
     *     message - the message text
     */
    public function action_send() {
        header('HTTP/1.0 500 Internal Error');
        $this->_widget->includeFileOnce('/lib/helpers/Index_MessageHelper.php');
        XG_App::includeFileOnce('/lib/XG_ContactHelper.php');
        if ($_SERVER['REQUEST_METHOD'] != 'POST') { throw new Exception('Not a POST (1317946116)'); }
        if (! User::isMember(XN_Profile::current())) { throw new Exception('Not a member (480178135)'); }
        if ($_POST['screenNames']) {
            XG_Cache::profiles($_POST['screenNames']);  // Prime the cache [Jon Aquino 2008-01-02]
            $friendScreenNames = $_POST['screenNames'];
        } else {
            $friendData = Index_MessageHelper::friendsOnNetwork($this->_user->screenName, $_POST['begin'], $_POST['end']);
            $friendScreenNames = Index_MessageHelper::removeAllFriendsOptOuts(User::screenNames($friendData['profiles']));
        }
        Index_MessageHelper::send($_POST['subject'], $_POST['message'], $friendScreenNames);
        header('HTTP/1.0 200 OK');
    }

}


