<?php

/**
 * Dispatches requests pertaining to invitations.
 */
class Index_InvitationController extends W_Controller {

    /** Whether the script is deliberately exiting with a 500 HTTP status code */
    private $exitingWith500 = false;

    /**
     * Runs code before each action.
     */
    protected function _before() {
        W_Cache::getWidget('main')->includeFileOnce('/lib/helpers/Index_InvitationFormHelper.php');
        W_Cache::getWidget('main')->includeFileOnce('/lib/helpers/Index_InvitationHelper.php');
        W_Cache::getWidget('main')->includeFileOnce('/lib/helpers/Index_MessageHelper.php');
        W_Cache::getWidget('main')->includeFileOnce('/lib/helpers/Index_InvitationMode.php');
        XG_HttpHelper::trimGetAndPostValues();
    }

    /**
     * Displays a form for sending invitations.
     *
     * Expected GET variables:
     *     sent - whether invitations were just sent
     *     noAddressesFound - whether the address import found 0 addresses
     *
     * @param $formToOpen string  (optional) which form to open: enterEmailAddresses, inviteFriends, webAddressBook, or emailApplication
     * @param $errors array  (optional) HTML error messages, optionally with keys field name
     */
    public function action_new($formToOpen = 'enterEmailAddresses', $errors = array()) {
        W_Cache::getWidget('main')->includeFileOnce('/lib/helpers/Index_MessageHelper.php');
        XG_SecurityHelper::redirectIfNotMember();
        if (! XG_App::canSendInvites($this->_user)) { return $this->redirectTo(xg_absolute_url('/')); }
        $this->showInvitationsSentMessage = $_GET['sent'];
        $this->showNoAddressesFoundMessage = $_GET['noAddressesFound'];
        $numFriends = Index_MessageHelper::numberOfFriendsAcrossNing($this->_user->screenName);
        $this->invitationArgs = array(
                'formToOpen' => $formToOpen,
                'errors' => $errors,
                'createUrl' => $this->_buildUrl('invitation', 'create'),
                'enterEmailAddressesButtonText' => xg_text('SEND_INVITATIONS'),
                'showBulkInvitationLink' => XG_SecurityHelper::userIsAdmin(),
                'inviteFriendsTitle' => xg_text('INVITE_FRIENDS'),
                'inviteFriendsDescription' => xg_text('INVITE_YOUR_FRIENDS_TO_APPNAME', XN_Application::load()->name),
                'friendDataUrl' => $this->_buildUrl('invitation', 'friendData', array('xn_out' => 'json')),
                'numFriends' => $numFriends,
				'messageParts' => XG_MessageHelper::getDefaultMessageParts(),
                'showInviteFriendsForm' => $numFriends > 0 && $numFriends > Index_MessageHelper::numberOfFriendsOnNetwork($this->_user->screenName));
    }

    /**
     * Outputs JSON for "friends" (each with screenName, fullName, thumbnailUrl,
     * and optional reasonToDisable) and "paginationHtml".
     *
     * Expected GET variables
     *     xn_out - "json";
     *     page - 1, 2, 3, ...
     */
    public function action_friendData() {
        W_Cache::getWidget('main')->includeFileOnce('/lib/helpers/Index_MessageHelper.php');
        $friendData = Index_MessageHelper::dataForFriendsAcrossNing($_GET['page']);
        $this->friends = $friendData['friends'];
        $this->paginationHtml = $friendData['paginationHtml'];
        $users = User::loadMultiple($friendData['screenNames']);
        $n = count($this->friends);
        for ($i = 0; $i < $n; $i++) {
            $user = $users[$this->friends[$i]['screenName']];
            if ($user && User::isMember($user)) {
                $this->friends[$i]['reasonToDisable'] = xg_text('ALREADY_MEMBER_OF_NETWORK');
            }
        }
    }

    /**
     * Processes the form for sending invitations.
     *
     * Expected POST variables:
     *
     *     form - "enterEmailAddresses"
     *     emailAddresses - email addresses separated by commas, semicolons, and whitespace
     *     message - optional message for the invitation
     *
     * or
     *
     *     form - "inviteFriends"
     *     allFriends - whether to send the message to all friends of the current user
     *     screenNames - JSON array of screen names of friends (if allFriends is false)
     *     inviteFriendsMessage - optional message for the invitation
     *
     * or
     *
     *     form - "webAddressBook"
     *     emailLocalPart - the part of the email address before the "@"
     *     emailDomain - the part of the email address after the "@"
     *     password - the password for the email address
     *
     * or
     *
     *     form - "emailApplication"
     *     file - a file containing CSV or VCF data
     */
    public function action_create() {
        XG_SecurityHelper::redirectIfNotMember();
        if (! XG_App::canSendInvites($this->_user)) { return $this->redirectTo(xg_absolute_url('/')); }
        if ($_SERVER['REQUEST_METHOD'] != 'POST') { return $this->redirectTo('new', 'invitation'); }
        switch ($_POST['form']) {

            case 'enterEmailAddresses':
                $result = Index_InvitationFormHelper::processEnterEmailAddressesForm();
                if ($result['errorHtml']) { return $this->forwardTo('new', 'invitation', array('enterEmailAddresses', array($result['errorHtml']))); }
                Index_InvitationFormHelper::send(array(
                        'inviteOrShare' => 'invite',
                        'contactList' => $result['contactList'],
                        'message' => $_POST['message']));
                $this->redirectTo('new', 'invitation', array('sent' => 1));
                break;

            case 'inviteFriends':
                $result = Index_InvitationFormHelper::processInviteFriendsForm();
                if ($result['errorHtml']) { return $this->forwardTo('new', 'invitation', array('inviteFriends', array($result['errorHtml']))); }
                Index_InvitationFormHelper::send(array(
                        'inviteOrShare' => 'invite',
                        'allFriends' => $result['allFriends'],
                        'contactList' => $result['contactList'],
                        'message' => $_POST['inviteFriendsMessage']));
                $this->redirectTo('new', 'invitation', array('sent' => 1));
                break;

            case 'webAddressBook':
                $result = Index_InvitationFormHelper::processWebAddressBookForm();
                if ($result['errorHtml']) { return $this->forwardTo('new', 'invitation', array('webAddressBook', array($result['errorHtml']))); }
                $this->redirectTo($result['target']);
                break;

            case 'emailApplication':
                $result = Index_InvitationFormHelper::processEmailApplicationForm();
                if ($result['errorHtml']) { return $this->forwardTo('new', 'invitation', array('emailApplication', array($result['errorHtml']))); }
                $this->redirectTo($result['target']);
                break;
        }
    }

    /**
     * Displays an AJAX-based form for editing the list of recipients for the invitation.
     *
     * Expected GET variables:
     *     contactListId - content ID of a ContactList object
     */
    public function action_editContactList() {
        XG_SecurityHelper::redirectIfNotMember();
        if (! XG_App::canSendInvites($this->_user)) { return $this->redirectTo(xg_absolute_url('/')); }
        if (! unserialize(ContactList::load($_GET['contactListId'])->my->contacts)) { return $this->redirectTo('new', 'invitation', array('noAddressesFound' => 1)); }
        $this->invitationArgs = array(
                'contactListId' => $_GET['contactListId'],
                'createWithContactListUrl' => $this->_buildUrl('invitation', 'createWithContactList', array('contactListId' => $_GET['contactListId'])),
                'cancelUrl' => $this->_buildUrl('invitation', 'new'),
                'inviteOrShare' => 'invite',
                'searchLabelText' => xg_text('SEARCH_FRIENDS_TO_INVITE'),
                'messageParts' => XG_MessageHelper::getDefaultMessageParts(),
                'submitButtonText' => xg_text('INVITE'));
    }

    /**
     * Processes the Contact List form.
     *
     * Expected GET variables:
     *     contactListId - content ID of a ContactList object
     *
     * Expected POST variables:
     *     contactListJson - a JSON array of contacts, each being an array with keys "name" and "emailAddress"
     *     message - optional message for the invitation
     */
    public function action_createWithContactList() {
        XG_SecurityHelper::redirectIfNotMember();
        if (! XG_App::canSendInvites($this->_user)) { return $this->redirectTo(xg_absolute_url('/')); }
        if ($_SERVER['REQUEST_METHOD'] != 'POST') { return $this->redirectTo('new', 'invitation'); }
        Index_InvitationFormHelper::processContactListForm('invite');
        $this->redirectTo('new', 'invitation', array('sent' => 1));
    }





    //#############################################################################
    // Common code. The actions below are shared between /main/invitation,
    // /groups/invitation, /events/invitation, and /main/sharing.
    //#############################################################################

    /**
     * Displays a form for choosing the method of inviting: Enter Email Address,
     * Web Address Book, or Email Application.
     *
     * @param $formToOpen string  which form to open: enterEmailAddresses, inviteFriends, webAddressBook, or emailApplication
     * @param $errors array  HTML error messages, optionally with keys field name
     * @param $createUrl string  the URL for processing the form for sending invitations
     * @param $enterEmailAddressesButtonText string  text for the submit button for the Enter Email Address form.
     * @param $showInviteFriendsForm string  whether to show the Invite Friends section
     * @param $inviteFriendsTitle string  title for the Invite Friends section
     * @param $inviteFriendsDescription string  description for the Invite Friends section
     * @param $friendDataUrl string  endpoint for retrieving friend info
     * @param $numFriends string  total number of friends
     * @param $showBulkInvitationLink boolean  whether to display the reusable invitation link
	 * @param $messageParts hash part_name => part_text. Used for warning user that his message contains potential spam.
     */
    public function action_chooseInvitationMethod($args) {
        XG_App::includeFileOnce('/lib/XG_ContactHelper.php');
        foreach ($args as $key => $value) { $this->{$key} = $value; }
        $this->enterEmailAddressesErrors = $this->formToOpen == 'enterEmailAddresses' ? $this->errors : array();
        $this->inviteFriendsErrors = $this->formToOpen == 'inviteFriends' ? $this->errors : array();
        $this->webAddressBookErrors = $this->formToOpen == 'webAddressBook' ? $this->errors : array();
        $this->emailApplicationErrors = $this->formToOpen == 'emailApplication' ? $this->errors : array();
        $this->emailDomains = Index_InvitationFormHelper::getEmailDomains();
        $this->showWebAddressBookForm = count($this->emailDomains) > 0;
        $this->emailDomains = array_merge(
                array('' => xg_text('SELECT_ELLIPSIS')),
                $this->emailDomains,
                array('(other)' => xg_text('OTHER_ELLIPSIS')));
        $importServices = XN_ContactImportService::listServices();
        $this->showEmailApplicationForm = $importServices['csv'] || $importServices['vcf'];
        $formDefaults = array();
        $emailParts = explode('@', $this->_user->email);
        if ($this->emailDomains[$emailParts[1]]) {
            $formDefaults['emailLocalPart'] = $emailParts[0];
            $formDefaults['emailDomain'] = $emailParts[1];
        }
        $this->form = new XNC_Form($formDefaults);

        if ($this->showBulkInvitationLink) {
            // BAZ-5429: Hide bulk invitation link if it can't be retrieved
            try {
                $this->bulkInvitationUrl = Index_InvitationHelper::getBulkInvitationUrl();
            } catch (Exception $e) {
                $this->showBulkInvitationLink = false;
            }
        }
    }

    /**
     * Displays a list of contacts that can be searched and sorted.
     *
     * @param $contactListId string  content ID of a ContactList object
     * @param $createWithContactListUrl string  the URL for processing the Contact List form
     * @param $cancelUrl string  the URL to go to when Cancel is pressed
     * @param $inviteOrShare string  the current context: "invite" or "share"
     * @param $searchLabelText string  text for the label beside the search field
     * @param $submitButtonText string  text for the button that submits the selected contacts
	 * @param $messageParts hash part_name => part_text. Used for warning user that his message contains potential spam.
     */
    public function action_contactList($args) {
        foreach ($args as $key => $value) { $this->{$key} = $value; }
        if (! $this->contactListId) { throw new Exception('No contact list specified (1657397187)'); }
        ContactList::cleanUp();
        $contactListObject = ContactList::load($this->contactListId);
        $this->contactList = unserialize($contactListObject->my->contacts);
        $count = count($this->contactList);
        for ($i = 0; $i < $count; $i++) {
            if (! $this->contactList[$i]['name']) {
                $this->contactList[$i]['name'] = Index_InvitationFormHelper::generateName($this->contactList[$i]['emailAddress']);
            }
        }
        usort($this->contactList, array($this, 'compareContactsByName'));
        $json = new NF_JSON(SERVICES_JSON_LOOSE_TYPE);
        $this->contactListJson = $json->encode($this->contactList);
        $this->cancelUrl = XG_HttpHelper::addParameters($this->_buildUrl('invitation', 'deleteContactList'), array('contactListId' => $this->contactListId, 'target' => $this->cancelUrl));
    }

    /**
     * Deletes a ContactList object.  ContactLists are temporary objects, used
     * when choosing recipients for an invitation..
     *
     * Expected GET variables:
     *     contactListId - content ID of a ContactList object
     *     target - the URL to redirect to afterwards
     */
    public function action_deleteContactList() {
        if (! $_GET['contactListId']) { throw new Exception('No contact list specified (1657397187)'); }
        if ($_SERVER['REQUEST_METHOD'] != 'POST') { throw new Exception('Not a POST (748876971)'); }
        try {
            XN_Content::delete(ContactList::load($_GET['contactListId']));
        } catch (Exception $e) {
            // Ignore [Jon Aquino 2007-10-25]
        }
        $this->redirectTo($_GET['target']);
    }

    /**
     * Returns -1, 0, or 1 depending on whether $a's name comes before, with, or after $b's name.
     *
     * @param $a  the first contact, with keys "name" and "emailAddress"
     * @param $b  the second contact, with keys "name" and "emailAddress"
     */
    private function compareContactsByName($a, $b) {
        return strcasecmp($a['name'], $b['name']);
    }

    /**
     * An interstitial page with spinner, shown during a long contact-list import.
     *
     * Expected GET parameters:
     *     jobId - a key identifying the job to monitor
     *     target - URL to go to after the import (contactListId will be appended)
     */
    public function action_waitForImport() {
        $this->jobId = $_GET['jobId'];
        $this->target = $_GET['target'];
    }

    /**
	 *  Checks message for strings that can be potentially marked as spam
     *
	 *  @param      $messageParts	hash	part-name:part-text. Text parts to check for spam.
	 *  @return     hash
	 *		status:			ok|warning|error		Not a spam/Potentially a spam/Probably a spam.
	 *		messageParts	hash<name:[text]>		List of exceprts
	 */
    public function action_checkMessageForSpam() {
		XG_App::includeFileOnce('/lib/XG_SpamHelper.php');
		$parts = $_REQUEST['messageParts'] ? json_decode($_REQUEST['messageParts']) : array();
		$this->messageParts = array();
		$count = 0;
		foreach ($parts as $name=>$value) {
			$bad = XG_SpamHelper::checkString($value);
			$this->messageParts[$name] = $bad;
			$count += count($bad);
		}
		$this->status = (0 == $count ? 'ok' : ($count <= 5 ? 'warning' : 'error'));
    }


    /**
     * Checks the status of the job that extracts email addresses.
     *
     * Expected GET parameters:
     *     xn_out - always "json"
     *     jobId - a key identifying the job to check on
     *
     * JSON output:
     *     complete - whether the job is finished
     *     contactListId - content ID of a ContactList object, if the job is finished
     */
    public function action_checkImport() {
        if ($_SERVER['REQUEST_METHOD'] != 'POST') { throw new Exception('Not a POST (1013274105)'); }
        $result = XN_ContactImportResult::load($_GET['jobId']);
        if (Index_InvitationHelper::isErrorArray($result)) { throw new Exception(Index_InvitationHelper::errorMessage(key($result))); }
        $this->complete = $result->status == XN_ContactImportResult::COMPLETE;
        if ($this->complete) { $this->contactListId = ContactList::create(Index_InvitationFormHelper::importedContactsToContactList(Index_InvitationFormHelper::allImportedContacts($result)))->id; }
    }

    /**
     * Sends messages to the specified people.
     *
     * Expected POST variables:
     *     inviteOrShare - the current context: "invite" or "share"
     *     groupId - the content ID of the associated Group, or null if none.
     *     eventId - the content ID of the associated Event, or null if none.
     *     contactList - JSON array of contacts, each an array with keys "name" and "emailAddress"
     *     friendStart - (if contactList is not specified) inclusive start index for a friend query
     *     friendEnd - (if contactList is not specified) exclusive end index for a friend query
     *     message - optional custom message
     *     contentId - the associated content ID for share messages; null for invitations
     *     retry - whether to retry if errors occur
     */
    public function action_send() {
        header('HTTP/1.0 500 Internal Error');
        if ($_SERVER['REQUEST_METHOD'] != 'POST') { throw new Exception('Not a POST (1013274105)'); }
        if (! User::isMember(XN_Profile::current())) { throw new Exception('Not a member (1209116584)'); }
        if ($_POST['friendEnd']) {
            $this->_widget->includeFileOnce('/lib/helpers/Index_MessageHelper.php');
            $friendData = Index_MessageHelper::friendsAcrossNing($this->_user->screenName, $_POST['friendStart'], $_POST['friendEnd']);
            $contactList = Index_InvitationFormHelper::screenNamesToContactList(Index_MessageHelper::removeAllFriendsOptOuts(User::screenNames($friendData['profiles'])));
        } else {
            $json = new NF_JSON(SERVICES_JSON_LOOSE_TYPE);
            $contactList = $json->decode($_POST['contactList']);
        }
        Index_InvitationMode::get($_POST)
                ->send($contactList, $_POST['message'], $_POST['contentId'], array($this, 'onSendError'));
        if (! $this->exitingWith500) { header('HTTP/1.0 200 OK'); }
    }

    /**
     * Called when an error occurs in Index_InvitationMode->send().
     *
     * Expected POST variables:
     *     inviteOrShare - the current context: "invite" or "share"
     *     groupId - the content ID of the associated Group, or null if none.
     *     eventId - the content ID of the associated Event, or null if none.
     *     contactList - JSON array of contacts, each an array with keys "name" and "emailAddress"
     *     message - optional custom message
     *     contentId - the associated content ID for share messages; null for invitations
     *     retry - whether to retry if errors occur
     *
     * @param $failedContacts array  contacts that were not processed because of errors
     */
    public function onSendError($failedContacts) {
        if (! $_POST['retry']) { $this->exitWith500(); }
        $onSendErrorProper = new XG_FaultTolerantTask(array($this, 'exitWith500'));
        $onSendErrorProper->add(array($this, 'onSendErrorProper'), array($failedContacts));
        $onSendErrorProper->execute(null);
    }

    /**
     * Called when an error occurs in Index_InvitationMode->send().
     *
     * Expected POST variables:
     *     inviteOrShare - the current context: "invite" or "share"
     *     groupId - the content ID of the associated Group, or null if none.
     *     eventId - the content ID of the associated Event, or null if none.
     *     contactList - JSON array of contacts, each an array with keys "name" and "emailAddress"
     *     message - optional custom message
     *     contentId - the associated content ID for share messages; null for invitations
     *     retry - whether to retry if errors occur
     *
     * @param $failedContacts array  contacts that were not processed because of errors
     */
    public function onSendErrorProper($failedContacts) {
        Index_InvitationFormHelper::send(array(
                'inviteOrShare' => $_POST['inviteOrShare'],
                'groupId' => $_POST['groupId'],
                'eventId' => $_POST['eventId'],
                'contactList' => $failedContacts,
                'message' => $_POST['message'],
                'contentId' => $_POST['contentId'],
                'retry' => false));
    }

    /**
     * Terminates the current action, with a 500 HTTP status code.
     */
    public function exitWith500() {
        // Log, for manual recovery [Jon Aquino 2007-11-15]
        $json = new NF_JSON(SERVICES_JSON_LOOSE_TYPE);
        error_log('exitWith500 ' . XN_Profile::current()->screenName . ' ' . $json->encode($_POST));
        $this->exitingWith500 = true;
        header('HTTP/1.0 500 Internal Error');
        exit;
    }

}
