<?php

/**
 * Dispatches requests pertaining to sign-in and sign-up.
 */
class Index_AuthorizationController extends W_Controller {

    // A nice feature of this design is that it maintains the "target" URL in a GET parameter,
    // when you move between the sign-in and sign-up pages. [Jon Aquino 2007-09-22]

    /**
     * Runs code before each action.
     */
    protected function _before() {
        $this->_widget->includeFileOnce('/lib/helpers/Index_AuthorizationHelper.php');
        XG_HttpHelper::trimGetAndPostValues();
    }

    /**
     * Pokes a hole in the app-wide privacy mechanism.
     *
     * @param $action string  The name of the action
     * @return boolean  Whether to bypass the privacy mechanism for the given action
     */
    public function action_overridePrivacy($action) {
        if (in_array($action, array('signIn', 'doSignIn', 'signOut', 'ningId', 'invitationOnly', 'termsOfService', 'privacyPolicy', 'problemsSigningIn', 'requestPasswordReset', 'doRequestPasswordReset', 'passwordResetSent', 'editPassword', 'updatePassword'))) {
            return true;
        }
        if (! XG_App::appIsPrivate()) { return true; }
        if (in_array($action, array('signUp', 'doSignUp', 'signUpNingUser', 'doSignUpNingUser'))) {
            return $this->currentUserCanSeeSignUpPage();
        }
        // 'newProfile', 'createProfile'
        W_Cache::getWidget('main')->includeFileOnce('/lib/helpers/Index_InvitationHelper.php');
        return XN_Profile::current()->isLoggedIn() &&
                (Index_InvitationHelper::getAcceptedInvitation(XN_Profile::current()->screenName) || XG_App::allowJoinByAll());
    }

    /**
     * Displays a form for signing in.
     *
     * Expected GET variables:
     *     target - (optional) URL to go to after sign-in
     *     groupToJoin - (optional) URL of the group to make the user a member of
     *     emailAddress - (optional) initial value for the Email Address field
     *     invitationExpired - (optional) whether to display a message saying that the invitation has expired
     *
     * @param $errors array  (optional) HTML error messages, optionally keyed by field name
     */
    public function action_signIn($errors = array()) {
        $this->target = $_GET['target'] ? $_GET['target'] : xg_absolute_url('/');
        $this->groupToJoin = $_GET['groupToJoin'];
        $this->form = new XNC_Form(array('emailAddress' => $_GET['emailAddress']));
        $this->errors = $errors;
        $this->showSignUpLink = $this->currentUserCanSeeSignUpPage();
        $this->showInvitationExpiredMessage = $_GET['invitationExpired'];
    }

    /**
     * Returns whether sign-up links are visible to the current user.
     *
     * @return boolean  whether the user can access the sign-up page
     */
    private function currentUserCanSeeSignUpPage() {
        return ! XG_App::appIsPrivate() || $this->associatedInvitation() || XG_App::allowJoinByAll();
    }

    /**
     * Processes the form for signing in.
     *
     * Expected GET variables:
     *     target - (optional) URL to go to after sign-in
     *     groupToJoin - (optional) URL of the group to make the user a member of
     *
     * Expected POST variables:
     *     emailAddress - email or username
     *     password - Ning password
     */
    public function action_doSignIn() {
        $this->doSignInProper('signIn', $_POST['emailAddress'], $_POST['password'], $_GET['target'], $_GET['groupToJoin']);
    }

    /**
     * Processes the form for signing in.
     *
     * @param $formAction string  the name of the action that displays the form
     * @param $emailAddress string  email or username
     * @param $password string  Ning password
     * @param $target string  (optional) URL to go to after sign-in
     * @param $groupToJoin string - (optional) URL of the group to make the user a member of
     */
    private function doSignInProper($formAction, $emailAddress, $password, $target, $groupToJoin) {
        if ($_SERVER['REQUEST_METHOD'] != 'POST') { return $this->redirectTo($formAction, 'authorization', array('target' => $target, 'groupToJoin' => $groupToJoin)); }
        if (! $emailAddress) {
            return $this->forwardOrRedirect($formAction, $emailAddress, $target, $groupToJoin, array('emailAddress' => xg_html('PLEASE_ENTER_EMAIL_ADDRESS')));
        }
        if (! $password) {
            return $this->forwardOrRedirect($formAction, $emailAddress, $target, $groupToJoin, array('password' => xg_html('PLEASE_ENTER_PASSWORD')));
        }
        if (is_array($result = XN_Profile::signIn($emailAddress, $password, array('max-age' => 2 * 365 * 24 * 60 * 60)))) {
            return $this->forwardOrRedirect($formAction, $emailAddress, $target, $groupToJoin,
                    Index_AuthorizationHelper::errorMessage(key($result), $formAction, $this->associatedInvitation(), $target, $groupToJoin));
        }
        $this->forwardOrRedirect($formAction, $emailAddress, $target, $groupToJoin);
    }

    /**
     * Logs the user out. In this action, we do any necessary clean up,
     * such as clearing invitation cookies, and setting a special 'just logged out'
     * cookie, and redirect to the homepage. The homepage, if it notices the
     * special 'just logged out' cookie, unsets that cookie and displays the
     * "you have just signed out" message. (Putting the just-signed-out status
     * in a cookie keeps it out of the URL and potential bookmarks, etc.
     *
     * Expected GET variables:
     *     target - (optional) URL to go to after sign-out
     */
    public function action_signOut() {
        if (! XG_SecurityHelper::checkCsrfTokenInPost() && ! XG_SecurityHelper::checkCsrfTokenInGet()) { throw new Exception('Not allowed (950700940)'); }
        // Set the just-logged-out cookie
        XG_App::setLogoutCookie(true); // TODO: is the just-logged-out cookie used anymore? [Jon Aquino 2008-04-21]
        XN_Profile::signOut();
        $target = $_GET['target'] ? $_GET['target'] : xg_absolute_url('/');
        header('Location: ' . $target);
    }

    /**
     * Displays a form for signing up someone with a Ning account.
     *
     * Expected GET variables:
     *     target - (optional) URL to go to after sign-in
     *     groupToJoin - (optional) URL of the group to make the user a member of
     *     emailAddress - (optional) initial value for the Email Address field, or null to use that of the current user (if any)
     *
     * @param $errors array  (optional) HTML error messages, optionally keyed by field name
     */
    public function action_signUpNingUser($errors = array()) {
        $this->target = $_GET['target'] ? $_GET['target'] : xg_absolute_url('/');
        $this->invitation = $this->associatedInvitation();
        $this->groupToJoin = $_GET['groupToJoin'];
        $this->form = new XNC_Form(array('emailAddress' => $_GET['emailAddress'] ? $_GET['emailAddress'] : XN_Profile::current()->email));
        $this->errors = $errors;
    }

    /**
     * Processes the form for signing up someone with a Ning account.
     *
     * Expected GET variables:
     *     target - (optional) URL to go to after sign-in
     *     groupToJoin - (optional) URL of the group to make the user a member of
     *
     * Expected POST variables:
     *     emailAddress - email or username
     *     password - Ning password
     */
    public function action_doSignUpNingUser() {
        $this->doSignInProper('signUpNingUser', $_POST['emailAddress'], $_POST['password'], $_GET['target'], $_GET['groupToJoin']);
    }

    /**
     * Displays a form for signing up.
     *
     * Expected GET variables:
     *     target - (optional) URL to go to after sign-up
     *     groupToJoin - (optional) URL of the group to make the user a member of
     *     emailAddress - (optional) initial value for the Email Address field
     *
     * @param $errors array  (optional) HTML error messages, optionally keyed by field name
     */
    public function action_signUp($errors = array()) {
        $this->target = $_GET['target'] ? $_GET['target'] : xg_absolute_url('/');
        $this->invitation = $this->associatedInvitation();
        $this->groupToJoin = $_GET['groupToJoin'];
        $this->form = new XNC_Form(array('emailAddress' => $_GET['emailAddress']));
        $this->errors = $errors;
        $this->captcha = XN_Auth_Captcha::create();
    }

    /**
     * Processes the form for signing up
     *
     * Expected GET variables:
     *     target - (optional) URL to go to after sign-in
     *     groupToJoin - (optional) URL of the group to make the user a member of
     *
     * Expected POST variables:
     *     emailAddress - email or username
     *     password - Ning password
     */
    public function action_doSignUp() {
        if ($_SERVER['REQUEST_METHOD'] != 'POST') { $this->redirectTo('signUp', 'authorization', array('target' => $_GET['target'], 'groupToJoin' => $_GET['groupToJoin'])); return; }
        if (true === XN_Profile::signIn($_POST['emailAddress'], $_POST['password'], array('max-age' => 2 * 365 * 24 * 60 * 60))) {
            return $this->forwardOrRedirect('signIn', $_POST['emailAddress'], $_GET['target'], $_GET['groupToJoin']);
        }
        $errors = array();
        XG_App::includeFileOnce('/lib/XG_ValidationHelper.php');
        if (! $_POST['emailAddress']) {
            $errors['emailAddress'] = xg_html('PLEASE_ENTER_EMAIL_ADDRESS');
        } elseif (! XG_ValidationHelper::isValidEmailAddress($_POST['emailAddress'])) {
            $errors['emailAddress'] = xg_html('EMAIL_NOT_VALID');
        }
        if (! $_POST['password']) {
            $errors['password'] = xg_html('PLEASE_ENTER_PASSWORD');
        } elseif (! $_POST['passwordConfirmation']) {
            $errors['passwordConfirmation'] = xg_html('PLEASE_ENTER_PASSWORD_AGAIN');
        } elseif ($_POST['password'] != $_POST['passwordConfirmation']) {
            $errors['passwordConfirmation'] = xg_html('PASSWORDS_DO_NOT_MATCH');
        }
        if (! $_POST['captchaValue']) {
            $errors['captchaValue'] = xg_html('PLEASE_ENTER_CODE');
        }
        if ($errors) {
            return $this->forwardOrRedirect('signUp', $_POST['emailAddress'], $_GET['target'], $_GET['groupToJoin'], $errors);
        }
        $captcha = XN_Auth_Captcha::create($_POST['captchaToken']);
        $captcha->value = $_POST['captchaValue'];
        $profile = XN_Profile::create($_POST['emailAddress'], $_POST['password']);
        if (is_array($result = $profile->save($captcha))) {
            return $this->forwardOrRedirect('signUp', $_POST['emailAddress'], $_GET['target'], $_GET['groupToJoin'],
                    Index_AuthorizationHelper::errorMessage(key($result), 'signUp', $this->associatedInvitation(), $_GET['target'], $_GET['groupToJoin']));
        }
        if (is_array($result = XN_Profile::signIn($_POST['emailAddress'], $_POST['password'], array('max-age' => 2 * 365 * 24 * 60 * 60)))) {
            return $this->forwardOrRedirect('signUp', $_POST['emailAddress'], $_GET['target'], $_GET['groupToJoin'],
                    Index_AuthorizationHelper::errorMessage(key($result), 'signUp', $this->associatedInvitation(), $_GET['target'], $_GET['groupToJoin']));
        }
        $this->forwardOrRedirect('signUp', $_POST['emailAddress'], $_GET['target'], $_GET['groupToJoin']);
    }

    /**
     * Displays a description of what a Ning ID is.
     *
     * Expected GET variables:
     *     previousUrl - (optional) target for the Back link
     */
    public function action_ningId() {
        $this->previousUrl = $_GET['previousUrl'] ? $_GET['previousUrl'] : xg_absolute_url('/');
    }

    /**
     * Displays a message saying that an invitation is required.
     */
    public function action_invitationOnly() {
        if (! $this->_user->isLoggedIn()) {
            $this->redirectTo(xg_absolute_url('/'));
            return;
        }
    }

    /**
     * Displays the full text of the Ning Terms of Service
     *
     * Expected GET variables:
     *     previousUrl - (optional) target for the Back link
     */
    public function action_termsOfService() {
        $this->previousUrl = $_GET['previousUrl'] ? $_GET['previousUrl'] : xg_absolute_url('/');
        $this->privacyPolicyUrl = $this->_buildUrl('authorization', 'privacyPolicy', array('previousUrl' => $this->previousUrl));
        $this->networkNameHtml = xnhtmlentities(XN_Application::load()->name);
        $this->hasCustomTermsOfService = $this->_widget->config['plugin_termsOfService'];
    }

    /**
     * Displays the full text of the Ning Privacy Policy
     *
     * Expected GET variables:
     *     previousUrl - (optional) target for the Back link
     */
    public function action_privacyPolicy() {
        $this->previousUrl = $_GET['previousUrl'] ? $_GET['previousUrl'] : xg_absolute_url('/');
        $this->termsOfServiceUrl = $this->_buildUrl('authorization', 'termsOfService', array('previousUrl' => $this->previousUrl));
        $this->networkNameHtml = xnhtmlentities(XN_Application::load()->name);
        $this->hasCustomPrivacyPolicy = $this->_widget->config['plugin_privacyPolicy'];
    }

    /**
     * Displays troubleshooting tips for sign-in and sign-up.
     *
     * Expected GET variables:
     *     previousUrl - (optional) target for the Back link
     */
    public function action_problemsSigningIn() {
        $this->previousUrl = $_GET['previousUrl'] ? $_GET['previousUrl'] : xg_absolute_url('/');
    }

    /**
     * Displays a form for requesting a reset-password email.
     *
     * Expected GET variables:
     *     previousUrl - (optional) target for the Back link
     *
     * @param $errors array  (optional) HTML error messages, optionally keyed by field name
     */
    public function action_requestPasswordReset($errors = array()) {
        $this->previousUrl = $_GET['previousUrl'] ? $_GET['previousUrl'] : xg_absolute_url('/');
        $this->form = new XNC_Form(array('previousUrl' => $this->previousUrl));
        $this->errors = $errors;
    }

    /**
     * Processes the form for requesting a reset-password email.
     *
     * Expected POST variables:
     *     emailAddress - email or username
     *     previousUrl - (optional) target for the Back link
     */
    public function action_doRequestPasswordReset() {
        if ($_SERVER['REQUEST_METHOD'] != 'POST') { $this->redirectTo('requestPasswordReset', 'authorization'); return; }
        if (! $_POST['emailAddress']) {
            return $this->forwardTo('requestPasswordReset', 'authorization', array(array('emailAddress' => xg_html('PLEASE_ENTER_EMAIL_ADDRESS'))));
        }
        try {
            if (! XN_Profile::load($_POST['emailAddress'])) {
                return $this->forwardTo('requestPasswordReset', 'authorization', array(array('emailAddress' => xg_html('NO_ACCOUNT_WITH_EMAIL_X', xnhtmlentities($_POST['emailAddress'])))));
            }
        } catch (Exception $e) {
            return $this->forwardTo('requestPasswordReset', 'authorization', array(array('emailAddress' => xg_html('NO_ACCOUNT_WITH_EMAIL_X', xnhtmlentities($_POST['emailAddress'])))));
        }

        $profile = XG_Cache::profiles($_POST['emailAddress']);
        $fullName = XG_UserHelper::getFullName($profile);
        $app = XN_Application::load();
        $appName = $app->name;

        $messageSubject = xg_text('RESET_PASSWORD_EMAIL_SUBJECT');
        $messageTemplate = xg_text('RESET_PASSWORD_EMAIL_BODY', $fullName, $appName);

        if (is_array($result = XN_Profile::resetPassword($_POST['emailAddress'], $messageSubject, $messageTemplate, $this->_buildUrl('authorization', 'editPassword')))) {
            return $this->forwardTo('requestPasswordReset', 'authorization', array(
                    Index_AuthorizationHelper::errorMessage(key($result), 'requestPasswordReset', $this->associatedInvitation())));
        }
        $this->redirectTo('passwordResetSent', 'authorization', array('previousUrl' => $_POST['previousUrl']));
    }

    /**
     * Displays a message saying that the password reset email has been sent.
     *
     * Expected GET variables:
     *     previousUrl - (optional) target for the Back link
     *
     * @param $errors array  (optional) HTML error messages, optionally keyed by field name
     */
    public function action_passwordResetSent($errors = array()) {
        $this->previousUrl = $_GET['previousUrl'] ? $_GET['previousUrl'] : xg_absolute_url('/');
    }

    /**
     * Displays a form for editing one's password.
     *
     * @param $errors array  (optional) HTML error messages, optionally keyed by field name
     */
    public function action_editPassword($errors = array()) {
        XG_SecurityHelper::redirectToSignInPageIfSignedOut();
        $this->form = new XNC_Form();
        $this->errors = $errors;
    }

    /**
     * Processes the form for editing one's password.
     *
     * Expected POST variables:
     *     password - the new password
     */
    public function action_updatePassword() {
        XG_SecurityHelper::redirectToSignInPageIfSignedOut();
        if ($_SERVER['REQUEST_METHOD'] != 'POST') { $this->redirectTo('editPassword', 'authorization'); return; }
        if (! $_POST['password']) {
            return $this->forwardOrRedirect('editPassword', null, null, null, array('password' => xg_html('CHOOSE_NEW_PASSWORD')));
        }
        $this->_user->password = $_POST['password'];
        if (is_array($result = $this->_user->save())) {
            return $this->forwardOrRedirect('editPassword', null, null, null, Index_AuthorizationHelper::errorMessage(key($result), 'editPassword', $this->associatedInvitation()));
        }
        $this->forwardOrRedirect('editPassword', XN_Profile::current()->email, null, null);
    }

    /**
     * Displays a form for filling out one's profile.
     *
     * Expected GET variables:
     *     target - (optional) URL to go to after sign-in
     *     newNingUser - "1" if the user's Ning account was just created
     *     groupToJoin - (optional) URL of the group to make the user a member of
     *
     * @param $errors array  (optional) HTML error messages, optionally keyed by field name
     */
    public function action_newProfile($errors = array()) {
        XG_SecurityHelper::redirectToSignUpPageIfSignedOut();
        $this->errors = $errors;
        $this->target = $_GET['target'] ? $_GET['target'] : xg_absolute_url('/');
        $this->newNingUser = $_GET['newNingUser'];
        $this->groupToJoin = $_GET['groupToJoin'];
        $this->showAboutSection = XG_UserHelper::getGender($this->_user) || XG_UserHelper::getBirthdate($this->_user) || XG_UserHelper::getLocation($this->_user) || XG_UserHelper::getCountry($this->_user);
    }

    /**
     * Processes the form for filling out one's profile.
     *
     * Expected GET variables:
     *     target - (optional) URL to go to after sign-in
     *     newNingUser - "1" if the user's Ning account was just created
     *     groupToJoin - (optional) URL of the group to make the user a member of
     *
     * Expected POST variables:
     *     fullName - display name
     *     photo - (optional) uploaded avatar image
     *     aboutQuestionsShown - Y to save the gender, birthdate, location, and country
     *     gender - (optional) (m)ale or (f)emale
     *     birthdateMonth - (optional) 1 for January, etc.
     *     birthdateDay - (optional) 1-31
     *     birthdateYear - (optional) four-digit year
     *     location - (optional) city name
     *     country - (optional) 2-letter country code, e.g., AU
     *     TODO: Document POST variables for custom profile questions
     */
    public function action_createProfile() {
        if (! $this->_user->isLoggedIn()) { error_log('BAZ-7028 @ action_createProfile() @ Current user: ' . XN_Profile::current()->screenName . ' @ Current URL: ' . XG_HttpHelper::currentURL() . ' @ Referrer: ' . $_SERVER['HTTP_REFERER']); }
        XG_SecurityHelper::redirectToSignUpPageIfSignedOut();
        if ($_SERVER['REQUEST_METHOD'] != 'POST') { return $this->redirectTo('newProfile', 'authorization', array('target' => $_GET['target'], 'newNingUser' => $_GET['newNingUser'], 'groupToJoin' => $_GET['groupToJoin'])); }
        $this->_widget->includeFileOnce('/lib/helpers/Index_ProfileInfoFormHelper.php');
        $this->_widget->includeFileOnce('/lib/helpers/Index_MembershipHelper.php');
        W_Cache::getWidget('profiles')->includeFileOnce('/lib/helpers/Profiles_ProfileQuestionFormHelper.php');
        if ($errors = array_merge(Index_ProfileInfoFormHelper::validateForm(), Profiles_ProfileQuestionFormHelper::validateForm())) {
            $this->forwardTo('newProfile', 'authorization', array($errors));
            return;
        }
        $user = User::loadOrCreate($this->_user);
        Index_ProfileInfoFormHelper::write($this->_user, $user, $_GET['newNingUser']);
        if (is_array($result = $this->_user->save())) {
            return $this->forwardTo('newProfile', 'authorization', array(
                    Index_AuthorizationHelper::errorMessage(key($result), 'newProfile', $this->associatedInvitation(), $_GET['target'], $_GET['groupToJoin'])));
        }
        Profiles_ProfileQuestionFormHelper::write($user);
        Index_MembershipHelper::onJoin($this->_user, $user); // Saves the User object [Jon Aquino 2007-09-24]
        if (! $_GET['target'] || XG_HttpHelper::isHomepage($_GET['target'])) {
            header('Location: ' . xg_absolute_url('/profiles'));
        } else {
            header('Location: ' . $_GET['target']);
        }
    }

    /**
     * Displays the form for editing basic profile info. Called by the Create Profile and Edit Profile pages.
     *
     * @param $errors array  (optional) HTML error messages, optionally keyed by field name
     * @param $showSimpleUploadField boolean  whether to show a file upload control instead of a BazelImagePicker
     * @param $showAboutSection boolean  whether to display the About Me section
     * @param $expandAboutSection boolean  whether to expand the About Me section to show the birthdate, country, and other fields
     * @param $indicateRequiredFields boolean  whether to add a special marker to required fields
     */
    public function action_profileInfoForm($errors, $showSimpleUploadField, $showAboutSection, $expandAboutSection, $indicateRequiredFields) {
        $this->errors = $errors;
        $this->_widget->includeFileOnce('/lib/helpers/Index_ProfileInfoFormHelper.php');
        $this->form = new XNC_Form(Index_ProfileInfoFormHelper::read($this->_user));
        $this->monthOptions = array(xg_text('MONTH'), xg_text('JANUARY'), xg_text('FEBRUARY'), xg_text('MARCH'), xg_text('APRIL'), xg_text('MAY'), xg_text('JUNE'), xg_text('JULY'), xg_text('AUGUST'), xg_text('SEPTEMBER'), xg_text('OCTOBER'), xg_text('NOVEMBER'), xg_text('DECEMBER'));
        $this->dayOptions = array('' => xg_text('DAY'));
        for ($i = 1; $i <= 31; $i++) { $this->dayOptions[$i] = $i; }
        $currentYear = date('Y', time());
        $this->yearOptions = array('' => xg_text('YEAR'));
        for ($i = $currentYear; $i >= $currentYear - 100 ; $i--) { $this->yearOptions[$i] = $i; }
        $this->countryOptions = array_merge(array('' => xg_text('SELECT')), Index_ProfileInfoFormHelper::popularCountries(), array('_2' => '----------------'), Index_ProfileInfoFormHelper::countries());
        $this->showSimpleUploadField = $showSimpleUploadField;
        $this->showAboutSection = $showAboutSection;
        $this->expandAboutSection = $expandAboutSection;
        if ($_SERVER['REQUEST_METHOD'] == 'POST') { $this->expandAboutSection = $_POST['aboutQuestionsShown'] == 'Y'; }
        $this->indicateRequiredFields = $indicateRequiredFields;
    }

    /**
     * Displays the footer for the sign-in and sign-up pages.
     *
     * @param $displayAvatars boolean  whether to display the avatars of some active users
     */
    public function action_footer($displayAvatars = true) {
        $minute = 60;
        // When adding a new argument, be sure to add it to the md5 hash below [Jon Aquino 2008-01-04]
        $this->setCaching(array(md5(implode(',', array(
                '/main/authorization/footer',
                $displayAvatars ? 'Y' : 'N')))), 30 * $minute);
        /* BAZ-4927: show some avatars in public networks */
        if ($displayAvatars && ! XG_App::appIsPrivate()) {
            W_Cache::getWidget('profiles')->includeFileOnce('/lib/helpers/Profiles_UserHelper.php');
            $this->profilesToDisplay = XG_Cache::profiles(Profiles_UserHelper::getActiveUsers(9));
        }
    }

    /**
     * Forwards or redirects to an appropriate URL.
     *
     * @param $formAction string  name of the action for the original form: signIn, signUp, signUpNingUser
     * @param $emailAddress string  the email address entered by the user
     * @param $target string  URL for the page to land on eventually, or null to go to the homepage
     * @param $groupToJoin string - (optional) URL of the group to make the user a member of
     * @param $errors array  HTML error messages, optionally keyed by field name, or null if no errors occurred
     */
    private function forwardOrRedirect($formAction, $emailAddress, $target, $groupToJoin, $errors = null) {
        $profile = XG_Cache::profiles($emailAddress);
        $args = Index_AuthorizationHelper::nextAction(array(
                'formAction' => $formAction,
                'emailAddress' => $emailAddress,
                'target' => $target,
                'groupToJoin' => $groupToJoin,
                'errors' => $errors,
                'signUpAllowed' => $this->currentUserCanSeeSignUpPage(),
                'isNingUser' => $profile,
                'isPending' => $profile && User::isPending($profile),
                'isMember' => $profile && User::isMember($profile)));
        if ($args[0] == 'forward') {
            return $this->forwardTo($args[1], 'authorization', array($args[2]));
        } elseif ($args[0] == 'redirect') {
            return $this->redirectTo($args[1], 'authorization', $args[2]);
        } else {
            throw new Exception('Assertion failed (1230500910) - ' . var_export($args, true));
        }
    }


    /**
     * Returns invitation metadata for the current user.
     *
     * @return array  an array of metadata for the invitation (including inviter),
     *         or null if no invitation is detected
     */
    private function associatedInvitation() {
        static $checked = false;
        static $invitation = null;
        if (! $checked) {
            $checked = true;
            W_Cache::getWidget('main')->includeFileOnce('/lib/helpers/Index_InvitationHelper.php');
            $invitation = Index_InvitationHelper::getUnusedInvitation($_GET['target']);
        }
        return $invitation;
    }


}


