<?php
/**
 * A category of discussions.
 */
class Category extends W_Model {

    /**
     * The title of the category. Be sure to escape this with xnhtmlentities() when displaying it.
     *
     * @var XN_Attribute::STRING
     * @rule length 1,200
     * @feature indexing text
     */
    public $title;
    const MAX_TITLE_LENGTH = 200;

    /**
     * The body of the category; scrubbed HTML.
     *
     * @var XN_Attribute::STRING optional
     * @rule length 1,100000
     * @feature indexing text
     */
    public $description;
    const MAX_DESCRIPTION_LENGTH = 100000;

    /**
     * Is this category public or private?
     *
     * @var XN_Attribute::STRING
     */
    public $isPrivate;

    /**
     * Content ID of the Group to which this object belongs
     *
     * @var XN_Attribute::STRING optional
     */
    public $groupId;

    /**
     * Which mozzle created this object?
     *
     * @var XN_Attribute::STRING
     * @feature indexing phrase
     */
    public $mozzle;

    /**
     * Space-delimited string of IDs from deleted categories whose Topics
     * have been moved to this category; this category's ID is also included, for convenience.
     * The maximum number of IDs is 100, which is the limit on an "in" filter.
     * The ID string "null" is used for Topics with null categoryIds.
     *
     * @var XN_Attribute::STRING optional
     */
    public $alternativeIds;

    /**
     * Position of this category
     *
     * @var XN_Attribute::NUMBER
     */
    public $order;

    /**
     * Whether people other than the network creator can post to this category
     *
     * @var XN_Attribute::STRING
     * @rule choice 1
     */
    public $membersCanAddTopics;
    public $membersCanAddTopics_choices = array('Y','N');

    /**
     * Whether members can add replies to discussions in this category.
     *
     * @var XN_Attribute::STRING
     * @rule choice 1
     */
    public $membersCanReply;
    public $membersCanReply_choices = array('Y','N');

/** xn-ignore-start 2365cb7691764f05894c2de6698b7da0 **/
// Everything other than instance variables goes below here

    /**
     * Constructs a new Category.
     *
     * @param $title string  The title of the category
     * @param $description string  The body of the category (HTML scrubbed of invalid tags)
     * @return W_Content  An unsaved Category
     */
    public function create($title = null, $description = null) {
        $category = W_Content::create('Category', $title, $description);
        $category->my->mozzle = W_Cache::current('W_Widget')->dir;
        $category->isPrivate = XG_App::appIsPrivate() || XG_GroupHelper::groupIsPrivate();
        return $category;
    }

    /**
     * Creates, updates, and deletes the app's Category objects, based on the given metadata.
     *
     * @param $metadata array  metadata for each Category object: id (optional), title, description, membersCanAddTopics, membersCanReply
     * @return array  The W_Content Category objects
     */
    public static function buildCategories($metadata) {
        $oldCategories = self::idsToObjects(self::findAll());
        $categories = array();
        $i = 0;
        foreach ($metadata as $categoryMetadata) {
            $i++;
            $category = $categoryMetadata['id'] ? W_Content::load($categoryMetadata['id']): Category::create();
            if ($category->type != 'Category') { xg_echo_and_throw('Type of ' . $category->id . ' is not Category'); }
            $category->my->order = $i;
            $category->title = self::cleanTitle($categoryMetadata['title']);
            $category->description = self::cleanDescription($categoryMetadata['description']);
            $category->my->membersCanAddTopics = $categoryMetadata['membersCanAddTopics'] ? 'Y' : 'N';
            $category->my->membersCanReply = $categoryMetadata['membersCanReply'] ? 'Y' : 'N';
            $category->my->alternativeIds = $categoryMetadata['alternativeIds'];
            if ($category->my->membersCanAddTopics == 'Y') { $category->my->membersCanReply = 'Y'; }
            $errors = $category->validate();
            if (count($errors)) { xg_echo_and_throw('Invalid content: ' . var_export($errors, true)); }
            $categories[] = $category;
            unset($oldCategories[$category->id]);
        }
        // Only save if no errors occurred [Jon Aquino 2007-03-27]
        foreach ($categories as $category) {
            $category->save();
            if (! $category->my->alternativeIds || ! in_array($category->id, explode(' ', $category->my->alternativeIds))) {
                $category->my->alternativeIds = trim($category->my->alternativeIds . ' ' . $category->id);
                $category->save();
            }
        }
        foreach ($oldCategories as $id => $category) {
            // Delete the category rather than the ID; otherwise the query cache won't get invalidated [Jon Aquino 2007-03-27]
            XN_Content::delete($category);
        }
        self::$categories = array();
        self::invalidateRecentTopicsCacheForAllCategories();
        return $categories;
    }

    /**
     * Returns the Category objects in their designated order.
     *
     * @param $includeOwnerOnlyCategories boolean  Whether to include categories to which only the owner can post
     * @return array  The Category XN_Content objects
     */
    public static function findAll($includeOwnerOnlyCategories = true) {
        if (is_null(self::$categories[$includeOwnerOnlyCategories]) || defined('UNIT_TESTING')) {
            $query = XN_Query::create('Content');
            if (XG_Cache::cacheOrderN() || (! XG_GroupHelper::inGroupContext())) {
                $query = XG_Query::create($query);
                $query->addCaching(XG_Cache::key('type', 'Category'));
            }
            $query->filter('owner');
            $query->filter('type', '=', 'Category');
            XG_GroupHelper::addGroupFilter($query);
            $query->order('my.order', 'asc', XN_Attribute::NUMBER);
            $widget = W_Cache::current('W_Widget');
            $query->filter('my.mozzle', '=', $widget->dir);
            if (! $includeOwnerOnlyCategories) { $query->filter('my.membersCanAddTopics', '=', 'Y'); }
            if (defined('UNIT_TESTING')) { $query->filter('my.test', '=', 'Y'); }
            self::$categories[$includeOwnerOnlyCategories] = $query->execute();
        }
        return self::$categories[$includeOwnerOnlyCategories];
    }

    /** A two-element array containing cached categories that (a) do not include owner-only categories (b) do include owner-only categories */
    private static $categories = array();

    /**
     * Constructs a mapping of ids to objects.
     *
     * @param $objects array  XN_Content objects from which to extract the ids
     * @return array  An array of id => object
     */
    private static function idsToObjects($objects) {
        $idsToObjects = array();
        foreach ($objects as $object) {
            $idsToObjects[$object->id] = $object;
        }
        return $idsToObjects;
    }

    /**
     * Scrubs, linkifies, and truncates the given description.
     *
     * @param $description string  The Category description
     * @return string  The cleaned up Category description
     */
    public static function cleanDescription($description) {
        $description = trim($description ? $description : '');
        return mb_substr(xg_linkify(xg_scrub($description)), 0, self::MAX_DESCRIPTION_LENGTH);
    }

    /**
     * Truncates the given title
     *
     * @param $title string  The Category title
     * @return string  The cleaned up Category title
     */
    public static function cleanTitle($title) {
        $title = trim($title ? $title : '');
        return mb_substr($title ? $title : xg_text('UNTITLED_CATEGORY'), 0, self::MAX_TITLE_LENGTH);
    }

    /**
     * Returns recent topics for the specified category.
     *
     * @param $category XN_Content|W_Content  the Category
     * @return array  XN_Content Topic objects
     */
    public static function recentTopics($category, $end = 3) {
        $query = XN_Query::create('Content');
        if (XG_Cache::cacheOrderN()) {
            $query = XG_Query::create($query);
            $query->setCaching(self::recentTopicsInvalidationKey($category->id));
        }
        self::addCategoryFilter($query, $category);
        $query->end($end);
        W_Cache::current('W_Widget')->includeFileOnce('/lib/helpers/Forum_Filter.php');
        return Forum_Filter::get('mostRecentlyUpdatedDiscussions')->execute($query);
    }

    /**
     * Filters the given topic query by category.
     *
     * @param $query XN_Query|XG_Query  the query to add the filter to
     * @param $category XN_Content|W_Content  the Category
     * @return XN_Query|XG_Query  the query
     */
    public static function addCategoryFilter($query, $category) {
        $filters = array();
        foreach (explode(' ', $category->my->alternativeIds) as $id) {
            $filters[] = XN_Filter('my.categoryId', '=', $id == 'null' ? null : $id);
        }
        return $query->filter(call_user_func_array(array('XN_Filter', 'any'), $filters));
    }

    /**
     * Clears the cache of recent topics for the specified category.
     *
     * @param $category XN_Content|W_Content  the Category (or null, which will do nothing)
     */
    public static function invalidateRecentTopicsCache($category) {
        if (! $category) { return; }
        if ($category->type != 'Category') { xg_echo_and_throw('Not a Category: ' . $category->type); }
        XG_Query::invalidateCache(self::recentTopicsInvalidationKey($category->id));
    }

    /**
     * Clears the cache of recent topics for all categories.
     */
    public static function invalidateRecentTopicsCacheForAllCategories() {
        foreach (self::findAll() as $category) {
            self::invalidateRecentTopicsCache($category);
        }
    }

    /**
     * Returns the key used to invalidate the recent-topics query for the specified category.
     *
     * @param $categoryId string  the ID of the Category
     * @return string  the invalidation key
     * @see David Sklar, "Query Caching", internal wiki
     * @see XG_Query#setCaching
     */
    protected static function recentTopicsInvalidationKey($categoryId) {
        if (! is_string($categoryId) && ! is_numeric($categoryId)) { xg_echo_and_throw('Not a string'); }
        return 'recent-topics-' . $categoryId;
    }

    /**
     * Returns the Category object with the given ID.
     *
     * @param $categoryId string  the ID of the Category - null if the Topic has not been assigned a Category,
     *     in which case we look for a Category with null as one of its alternativeIds
     * @return W_Content  the matching Category object, or null if it no longer exists
     */
    public static function find($categoryId) {
        if (! is_null($categoryId) && ! is_string($categoryId) && ! is_numeric($categoryId)) { xg_echo_and_throw('Not a string'); }
        foreach (self::findAll() as $category) {
            if (in_array($categoryId ? $categoryId : 'null', explode(' ', $category->my->alternativeIds))) { return $category; }
        }
        return null;
    }

    /**
     * Sets the topic's category
     *
     * @param $topic W_Content|XN_Content  The Topic
     * @param $categoryId string  The content ID of the Category
     */
    public static function setCategoryId($topic, $categoryId) {
        if (self::find($topic->my->categoryId) && $topic->my->categoryId != $categoryId) {
            self::$categoryIdsToInvalidateOnSave[] = $topic->my->categoryId;
        }
        $topic->my->categoryId = $categoryId;
    }

    /** IDs of categories for which to invalidate the recent topic caches the next time a Topic is saved. */
    private static $categoryIdsToInvalidateOnSave = array();

    /**
     * Called after a content object has been saved or before a content object has been deleted.
     *
     * @param $object mixed  The content object, an array, or possibly some other thing if the XN_Event API changes
     */
    public static function contentSavedOrDeleted($object) {
        if (is_array($object)) {
            foreach ($object as $o) { self::contentSavedOrDeleted($o); }
            return;
        }
        if (! ($object instanceof XN_Content || $object instanceof W_Content)) { return; }
        if ($object->type == 'Topic') {
            // Ensure W_Cache::current('W_Widget') returns the correct value [Jon Aquino 2007-03-29]
            W_Cache::push(W_Cache::getWidget($object->my->mozzle));
            self::invalidateRecentTopicsCache(self::find($object->my->categoryId));
            foreach (self::$categoryIdsToInvalidateOnSave as $categoryId) {
                self::invalidateRecentTopicsCache(self::find($categoryId));
            }
            self::$categoryIdsToInvalidateOnSave = array();
            W_Cache::pop(W_Cache::current('W_Widget'));
        }
    }

/** xn-ignore-end 2365cb7691764f05894c2de6698b7da0 **/

}

XN_Event::listen('xn/content/save/after', array('Category', 'contentSavedOrDeleted'));
XN_Event::listen('xn/content/delete/before', array('Category', 'contentSavedOrDeleted'));


