<?php
/** $Id: class.php 3530 2006-08-15 14:48:07Z andrew $
 *
 *  Useful functions related to Events.
 *
 **/
class Events_EventHelper {

    /** Indicates that a unit test is being run. */
    static public $isUnitTest = 1; // prevent accidential access. Call EventWidget::init()

    /** Number of queries. For unit testing. */
    static public $count = 0;

    /** Async jobs data. For unit testing. */
    static protected $_jobs = array();

    /**
     *  Cleans up a URL entered by the user.
     *  Returns an empty string if the URL is just "http://".
     *
     *  @param      $url   string
     *  @return     string the trimmed URL, with http:// prepended if missing.
     */
    public static function url($url) {
        $url = trim($url);
        if (preg_match('#^(https?://)?$#ui',$url)) {
            return '';
        }
        if (!preg_match('#^(https?|ftp)://#ui',$url)) {
            $url = 'http://'.$url;
        }
        return $url;
    }

    /**
     *  Converts the date from string to Unix timestamp
     *
     *  @param      $dateStr    string|int      Any of: Unix timestamp, YYYY-MM-DD, YYYY-MM-DD HH:II, or null (for the current time)
     *  @return     int                         Unix timestamp
     */
    public static function dateToTs($dateStr = null) {
        if ($dateStr === NULL) {
            return xg_date('U');
        }
        if (is_numeric($dateStr)) {
            return $dateStr;
        }
        if (!preg_match('/^(\d{4})-(\d\d)-(\d\d)(?: (\d\d):(\d\d))?$/u', $dateStr, $m)) {
            return 0;
        }
        return xg_mktime($m[4],$m[5],0,$m[2],$m[3],$m[1]);
    }

    /**
     *  Converts the date to a string with format YYYY-MM-DD HH:II
     *
     *  @param      $ts     int     Unix timestamp, or null for the current time
     *  @return     string          YYYY-MM-DD HH:II
     */
    public static function dateToStr($ts = null) {
        if ($ts === NULL) {
            $ts = time();
        }
        if (!is_numeric($ts)) {
            $ts = self::dateToTs($ts);
        }
        return xg_date('Y-m-d H:i',$ts);
    }

    /**
     *  Returns the string for filtering queries by type. Used with the "like" query filter.
     *
     *  @param      $type       string  The Event type
     *  @return     string              "like" argument for querying Event->my->eventType
     */
    public static function typeFilter($type) {
        return '#' . base64_encode($type) . '#';
    }

    /**
     *  Converts the internal format to a list of event-types.
     *
     *  @param      $type       string  Event-type string, such as that in Event->my->eventType
     *  @return     list<string>        Array of event types
     */
    public static function typeToList($type) {
        return $type ? array_map('base64_decode',explode('#',trim($type,'#'))) : array();
    }

    /**
     *  Converts the list of values to the internal format
     *
     *  @param      $list   list<string>    Array of event types
     *  @return     type                    Event-type string, such as that in Event->my->eventType
     */
    public static function listToType($list) {
        return '#' . join('#',array_map('base64_encode',array_unique($list))) . '#';
    }

    /**
     *  Converts a user-entered, comma-delimited list of event types
     *  to the internal format (for storing in the content store).
     *  Text format is the same as for tags.
     *
     *  @param      $text   string      Text separated by commas or quotes.
     *  @return     type                Event-type string, such as that in Event->my->eventType
     */
    public static function textToType($text) {
        return '#' . join('#',array_map('base64_encode',array_unique(XN_Tag::parseTagString($text)))) . '#';
    }

    /**
     *  Converts a status code to a value for passing in urls
     * (to make them more human-friendly)
     *
     *  @param      $rsvp   int     One of EventAttendee::XXX constants
     *  @return     string          A URL parameter value, e.g., might_attend
     */
    public static function rsvpToStr($rsvp) {
        switch ($rsvp) {
            case EventAttendee::ATTENDING:      return 'attending';
            case EventAttendee::MIGHT_ATTEND:   return 'might_attend';
            case EventAttendee::NOT_ATTENDING:  return 'not_attending';
            case EventAttendee::NOT_RSVP;       return 'not_rsvped';
            default:                            return 'unknown';
        }
    }

    /**
     *  Converts a URL parameter value to a status code.
     *
     *  @param      $rsvp   string  A URL parameter value, e.g., might_attend
     *  @return     int             One of EventAttendee::XXX constants
     */
    public static function strToRsvp($rsvp) { # string
        // These constants are also used in BroadcastEventMessageLink.js [Jon Aquino 2008-04-01]
        switch ($rsvp) {
            case 'attending':       return EventAttendee::ATTENDING;
            case 'might_attend':    return EventAttendee::MIGHT_ATTEND;
            case 'not_attending':   return EventAttendee::NOT_ATTENDING;
            case 'not_rsvped':      return EventAttendee::NOT_RSVP;
            default:                return 0;
        }
    }

    /**
     * Decrements the count for the given key, and removes it if it's become zero.
     *
     * @param   $arr    hash    keys and counts
     * @param   $idx    string  name of the key to decrement
     */
    public static function decrement(array& $arr, $idx) { # void
        if (!isset($arr[$idx])) {
            return;
        }
        if ($arr[$idx] > 1) {
            $arr[$idx]--;
        } else {
            unset($arr[$idx]);
        }
    }

    /**
     * Creates a content object with the given type.
     * Sets the mozzle and isPrivate attributes.
     *
     * @param   $type   string  The content type
     * @return  W_Content       The unsaved content object
     */
    public static function create($type) { # W_Content
        $object = W_Content::create($type);
        $object->my->mozzle     = W_Cache::getWidget('events')->dir;
        $object->isPrivate      = XG_App::appIsPrivate();
        return $object;
    }

    /**
     *  Updates an object's properties and saves it. Properties are automatically
     *  prefixed with my->.
     *
     *  @param      $command    Events_EventCommand Not used
     *  @param      $object     W_Content           The object to update
     *  @param      $props      hash                New properties; array values will be serialized
     *  @return     void
     */
    public static function update($command, W_Content $object, array $props) {
        // This function lives here unless we find some better place.
        foreach ($props as $k=>$v) {
            if ($k == 'title' || $k == 'description' || $k == 'isPrivate') {
                $object->$k = $v;
            } else {
                $object->my->$k = is_array($v) ? serialize($v) : $v;
            }
        }
        if (/*self::$isUnitTest && */$errs = $object->validate()) {
            var_dump($errs, $object->export());
            echo "<pre>";
            debug_print_backtrace();
            echo "</pre>";
        }
        $object->save();
    }

    /**
     *  Removes the object.
     *
     *  @param      $command    Events_EventCommand Not used.
     *  @param      $object     W_Content           The object to delete.
     *  @return     void
     **/
    public static function delete($command, W_Content $object) {
        W_Content::delete($object);
    }

    /**
     *  Creates a query for a given content type.
     *
     *  @param      $type       string      The content type
     *  @param      $method     string      Content or Search
     *  @return     XN_Query
     */
    public static function query($type, $method = 'Content') {
        switch($method) {
            case 'Content': $query = XN_Query::create('Content')->filter('owner')->filter('type', '=', $type); break;
            case 'Search':  $query = XN_Query::create('Search')->filter('type', 'like', $type); break;
            default:        throw new Exception('Assertion failed (1668909308)');
        }
        if (self::$isUnitTest) {
            $query->filter('my->test','=','Y');
            self::$count++;
        }
        return $query;
    }

    /**
     *  Creates and saves a new async job. Tasks will be executed in parallel.
     *  TASK_DEF is a list where the first item is a callback and the rest are positional parameters.
     *  Callbacks must have the format:
     *      array(CLASS,task_METHOD)    if method name does not start with "task_", an error is generated.
     *  task_METHOD signature:
     *      function( ..positional-params..) void
     *
     *  @param      $tasks   list<TASK_DEF>    List of tasks to run.
     *  @return     void
     */
    public static function createJob(array $tasks) {
        if (!$tasks) {
            throw new Exception("Empty job! (2359672867)");
        }
        if (self::$isUnitTest) {
            self::$_jobs[] = array('tasks' => $tasks);
        } else {
            $widget = W_Cache::getWidget('events');
            if (!$asyncKey = W_Cache::getWidget('events')->privateConfig['asyncKey']) {
                $widget->privateConfig['asyncKey'] = $asyncKey = md5(uniqid(rand(), true));
                $widget->saveConfig();
            }
            $job = XN_Job::create();
            foreach ($tasks as $task) {
                $job->addTask( XN_Task::create( $widget->buildUrl('index','asyncJob'), array('asyncKey'=> $asyncKey, 'task'=>$task) ) );
            }
            $job->save();
        }
    }

    /**
     *  Dispatches task. $args is typically $_REQUEST.
     *
     *  @param      $asyncKey	string                  Secret async key
     *  @param      $task       list<method, ..args..>  TASK_DEF (see above)
     *  @return     void
     */
    public static function dispatchJob($args) {
        if (!self::$isUnitTest) {
            $asyncKey = W_Cache::getWidget('events')->privateConfig['asyncKey'];
            if (!$asyncKey || $args['asyncKey'] != $asyncKey) {
                throw new Exception("asyncKey mismatch");
            }
        }
        $task = $args['task'];
        $callback = array_shift($task);
        if ( !is_array($callback) || count($callback)!=2 || !is_string($callback[0]) || !preg_match('/^task_\w+$/u',$callback[1]) ) {
            throw new Exception("Malformed task handler:".var_export($callback,TRUE));
        }
        # run task
        $start = microtime(true);
        call_user_func_array($callback, $task);
        error_log("asyncJob $callback[0]::$callback[1](".join(", ",$task)."): done in ".sprintf('%.4f',microtime(true)-$start));
    }

    /**
     * Executes all scheduled async jobs. For unit testing.
     */
    public static function _dispatchAllJobs() { # void
        while (count(self::$_jobs)) { // cannot use foreach, because jobs can create other jobs
            $job = array_shift(self::$_jobs);
            foreach ($job['tasks'] as $task) {
                self::dispatchJob(array('task' => $task));
            }
        }
    }

    /**
     *  Fills $keys from cache and returns the list of missed IDs. ID is any unique value.
     *
     *  @param      $keys   hash<id:cache-key>  Cache keys with arbitrary hash keys
     *  @return     list<string>                Hash keys for cache keys with no cache entries
     */
    public static function fillFromCache(array &$keys) {
        if (!XG_Cache::cacheOrderN() || self::$isUnitTest) {
            return array_keys($keys);
        }
        $res        = XN_Cache::get($keys);
        $missing    = array();
        foreach ($keys as $k=>$v) {
            if (isset($res[$v])) {
                $keys[$k] = $res[$v];
            } else {
                $missing[] = $k;
            }
        }
        return $missing;
    }

    /**
     *  The same as XN_Cache::put, but disabled during unit testing.
     *
     *  @return     void
     */
    public static function cachePut($key, $object, $labels) {
        if (XG_Cache::cacheOrderN() && !self::$isUnitTest) {
            XN_Cache::put($key, $object, $labels);
        }
    }
    /**
     *  The same as XN_Cache::insert, but disabled during unit testing
     *
     *  @return     void
     */
    public static function cacheInsert($key, $object, $labels) {
        if (XG_Cache::cacheOrderN() && !self::$isUnitTest) {
            XN_Cache::insert($key, $object, $labels);
        }
    }
}
?>
