<?php

/**
 * Dispatches requests pertaining to invitations.
 */
class Events_InvitationController extends W_Controller {

    /**
     * Constructor.
     *
     * @param   $widget     W_BaseWidget    The Events widget
     */
    public function __construct(W_BaseWidget $widget) {
        parent::__construct($widget);
        EventWidget::init();
    }

    /**
     * Runs code before each action.
     */
    protected function _before() {
        $this->_widget->includeFileOnce('/lib/helpers/Events_SecurityHelper.php');
        $this->_widget->includeFileOnce('/lib/helpers/Events_InvitationHelper.php');
        W_Cache::getWidget('main')->includeFileOnce('/lib/helpers/Index_InvitationFormHelper.php');
        XG_HttpHelper::trimGetAndPostValues();
    }

    /**
     * Displays a form for sending invitations.
     *
     * Expected GET variables:
     *     sent - whether invitations were just sent
     *     noAddressesFound - whether the address import found 0 addresses
     *     creatingEvent - whether the event has just been created
     *     eventId - the content ID of the associated Event.
     *
     * @param $formToOpen string  (optional) which form to open: enterEmailAddresses, inviteFriends, webAddressBook, or emailApplication
     * @param $errors array  (optional) HTML error messages, optionally with keys field name
     */
    public function action_new($formToOpen = 'enterEmailAddresses', $errors = array()) {
        W_Cache::getWidget('main')->includeFileOnce('/lib/helpers/Index_MessageHelper.php');
        $this->_widget->includeFileOnce('/lib/helpers/Events_TemplateHelper.php');
        XG_App::includeFileOnce('/lib/XG_ContactHelper.php');
        XG_SecurityHelper::redirectIfNotMember();
        $this->event = Event::byId($_GET['eventId']);
        if (! Events_SecurityHelper::currentUserCanSendInvites($this->event)) { return $this->redirectTo(xg_absolute_url('/')); }
        $this->showInvitationsSentMessage = $_GET['sent'];
        $this->showNoAddressesFoundMessage = $_GET['noAddressesFound'];
        $this->creatingEvent = $_GET['creatingEvent'];
        $numFriends = Index_MessageHelper::numberOfFriendsAcrossNing($this->_user->screenName);
        $this->invitationArgs = array(
                'formToOpen' => $formToOpen,
                'errors' => $errors,
                'createUrl' => $this->_buildUrl('invitation', 'create', array('eventId' => $this->event->id, 'creatingEvent' => $this->creatingEvent)),
                'enterEmailAddressesButtonText' => xg_text('SEND_INVITATIONS'),
                'inviteFriendsTitle' => xg_text('INVITE_FRIENDS'),
                'inviteFriendsDescription' => xg_text('INVITE_YOUR_FRIENDS_TO_EVENTNAME', $this->event->title),
                'friendDataUrl' => $this->_buildUrl('invitation', 'friendData', array('xn_out' => 'json', 'eventId' => $this->event->id)),
                'numFriends' => $numFriends,
				'messageParts' => Events_InvitationHelper::getMessageParts($this->event),
                'showInviteFriendsForm' => $numFriends > 0);
    }

    /**
     * Processes the form for sending invitations.
     *
     * Expected GET variables:
     *     creatingEvent - whether the event has just been created
     *     eventId - the content ID of the associated group.
     *
     * Expected POST variables:
     *
     *     form - "enterEmailAddresses"
     *     emailAddresses - email addresses separated by commas, semicolons, and whitespace
     *     message - optional message for the invitation
     *
     * or
     *
     *     form - "inviteFriends"
     *     allFriends - whether to send the message to all friends of the current user
     *     screenNames - JSON array of screen names of friends (if allFriends is false)
     *     inviteFriendsMessage - optional message for the invitation
     *
     * or
     *
     *     form - "webAddressBook"
     *     emailLocalPart - the part of the email address before the "@"
     *     emailDomain - the part of the email address after the "@"
     *     password - the password for the email address
     *
     * or
     *
     *     form - "emailApplication"
     *     file - a file containing CSV or VCF data
     */
    public function action_create() {
        XG_SecurityHelper::redirectIfNotMember();
        $event = Event::byId($_GET['eventId']);
        if (! Events_SecurityHelper::currentUserCanSendInvites($event)) { return $this->redirectTo(xg_absolute_url('/')); }
        if ($_SERVER['REQUEST_METHOD'] != 'POST') { return $this->redirectTo('new', 'invitation', array('eventId' => $event->id)); }
        switch ($_POST['form']) {

            case 'enterEmailAddresses':
                $result = Index_InvitationFormHelper::processEnterEmailAddressesForm();
                if ($result['errorHtml']) { return $this->forwardTo('new', 'invitation', array('enterEmailAddresses', array($result['errorHtml']))); }
                Index_InvitationFormHelper::send(array(
                        'inviteOrShare' => 'invite',
                        'eventId' => $event->id,
                        'contactList' => $result['contactList'],
                        'message' => $_POST['message']));
                if ($_GET['creatingEvent']) {
                    $this->redirectTo('show', 'event', array('id' => $event->id));
                } else {
                    $this->redirectTo('new', 'invitation', array('sent' => 1, 'eventId' => $event->id));
                }
                break;

            case 'inviteFriends':
                $result = Index_InvitationFormHelper::processInviteFriendsForm();
                if ($result['errorHtml']) { return $this->forwardTo('new', 'invitation', array('inviteFriends', array($result['errorHtml']))); }
                Index_InvitationFormHelper::send(array(
                        'inviteOrShare' => 'invite',
                        'eventId' => $event->id,
                        'allFriends' => $result['allFriends'],
                        'contactList' => $result['contactList'],
                        'message' => $_POST['inviteFriendsMessage']));
                if ($_GET['creatingEvent']) {
                    $this->redirectTo('show', 'event', array('id' => $event->id));
                } else {
                    $this->redirectTo('new', 'invitation', array('sent' => 1, 'eventId' => $event->id));
                }
                break;

            case 'webAddressBook':
                $result = Index_InvitationFormHelper::processWebAddressBookForm();
                if ($result['errorHtml']) { return $this->forwardTo('new', 'invitation', array('webAddressBook', array($result['errorHtml']))); }
                if ($_GET['creatingEvent']) {
                    $result['target'] = XG_HttpHelper::addParameter($result['target'],'creatingEvent', 1);
                }
                $this->redirectTo($result['target']);
                break;

            case 'emailApplication':
                $result = Index_InvitationFormHelper::processEmailApplicationForm();
                if ($result['errorHtml']) { return $this->forwardTo('new', 'invitation', array('emailApplication', array($result['errorHtml']))); }
                if ($_GET['creatingEvent']) {
                    $result['target'] = XG_HttpHelper::addParameter($result['target'],'creatingEvent', 1);
                }
                $this->redirectTo($result['target']);
                break;
        }
    }

    /**
     * Displays an AJAX-based form for editing the list of recipients for the invitation.
     *
     * Expected GET variables:
     *     contactListId - content ID of a ContactList object
     *     creatingEvent - whether the event has just been created
     */
    public function action_editContactList() {
        XG_SecurityHelper::redirectIfNotMember();
        $this->event = Event::byId($_GET['eventId']);
        if (! Events_SecurityHelper::currentUserCanSendInvites($this->event)) { return $this->redirectTo(xg_absolute_url('/')); }
        if (! unserialize(ContactList::load($_GET['contactListId'])->my->contacts)) { return $this->redirectTo('new', 'invitation', array('noAddressesFound' => 1, 'eventId' => $this->event->id)); }
        $this->invitationArgs = array(
                'contactListId' => $_GET['contactListId'],
                'createWithContactListUrl' => $this->_buildUrl('invitation', 'createWithContactList', array('contactListId' => $_GET['contactListId'], 'eventId' => $this->event->id, 'creatingEvent' => $_GET['creatingEvent'])),
                'cancelUrl' => $this->_buildUrl('invitation', 'new', array('eventId' => $this->event->id)),
                'inviteOrShare' => 'invite',
                'messageParts' => Events_InvitationHelper::getMessageParts($this->event),
                'searchLabelText' => xg_text('SEARCH_FRIENDS_TO_INVITE'),
                'submitButtonText' => xg_text('INVITE'));
    }

    /**
     * Processes the Contact List form.
     *
     * Expected GET variables:
     *     contactListId - content ID of a ContactList object
     *     creatingEvent - whether the event has just been created
     *
     * Expected POST variables:
     *     contactListJson - a JSON array of contacts, each being an array with keys "name" and "emailAddress"
     *     message - optional message for the invitation
     */
    public function action_createWithContactList() {
        XG_SecurityHelper::redirectIfNotMember();
        $event = Event::byId($_GET['eventId']);
        if (! Events_SecurityHelper::currentUserCanSendInvites($event)) { return $this->redirectTo(xg_absolute_url('/')); }
        if ($_SERVER['REQUEST_METHOD'] != 'POST') { return $this->redirectTo('new', 'invitation', array('eventId' => $event->id)); }
        Index_InvitationFormHelper::processContactListForm('invite', null, $event->id);
        if ($_GET['creatingEvent']) {
            $this->redirectTo('show', 'event', array('id' => $event->id));
        } else {
            $this->redirectTo('new', 'invitation', array('sent' => 1, 'eventId' => $event->id));
        }
    }

    /**
     * Outputs JSON for "friends" (each with screenName, fullName, thumbnailUrl,
     * and optional reasonToDisable) and "paginationHtml".
     *
     * Expected GET variables
     *     xn_out - "json";
     *     page - 1, 2, 3, ...
     *     eventId - the content ID of the associated event, or null if none.
     */
    public function action_friendData() {
        W_Cache::getWidget('main')->includeFileOnce('/lib/helpers/Index_MessageHelper.php');
        $friendData = Index_MessageHelper::dataForFriendsAcrossNing($_GET['page']);
        $this->friends = $friendData['friends'];
        $this->paginationHtml = $friendData['paginationHtml'];
        $friendScreenNames = array();
        foreach ($friendData['friends'] as $friend) {
            $friendScreenNames[] = $friend['screenName'];
        }
        $rsvpedScreenNames = array();
        foreach (array_chunk($friendScreenNames, 100) as $chunk) {
            $rsvpedScreenNames += EventAttendee::getRsvpedScreenNames($chunk, $_GET['eventId']);
        }
        $n = count($this->friends);
        for ($i = 0; $i < $n; $i++) {
            if ($rsvpedScreenNames[$this->friends[$i]['screenName']]) {
                $this->friends[$i]['reasonToDisable'] = xg_text('ALREADY_RSVPED');
            }
        }
    }

}
