<?php

/**
 * This class tracks the code revisions in the app and makes possible
 * taking actions when code in the app or (eventually) in individual
 * widgets has changed
 *
 * @ingroup XG
 */
class XG_Version {
    /**
     * The full path of the code which we hope will contain some parseable
     *   branch information
     */
    protected static $release = '$HeadURL: svn://app.svn.ninginc.com/bazel/tags/3.1.8/lib/XG_Version.php $';

    /**
     * The latest revision of the app code. This should be updated on every submit
     */
    protected static $revision = '$Revision: 4766 $'; /*1210301068*/

    /**
     * The appropriate string value for the latest app revision
     *
     * @return string
     */
    public static function currentCodeVersion() {
        if (preg_match('@/branches/([0-9][0-9\.]+)/lib/XG_Version.php@u', self::$release, $matches)) {
            $release = $matches[1] . ':';
        } elseif (preg_match('@/tags/([0-9][0-9\.]+)/lib/XG_Version.php@u', self::$release, $matches)) {
            $release = $matches[1] . ':';
        } else {
            $release = '';
        }
        $revision = intval(mb_substr(self::$revision, 2+mb_strpos(self::$revision, ':')));
        return $release . $revision;
    }

    /**
     *  Is the app-wide code version newer than the specified version?
     *
     *  Used in XG_App to detect code updates
     */
    protected static function codeIsNewerThan($haveString) {
        return self::olderThan($haveString, self::currentCodeVersion());
    }

    /**
     * Compare two version strings of the form release:revision first by release,
     *   then by revision
     *  Returns < 0 if $stringOne is less than $stringTwo; > 0 if $stringOne is greater than $stringTwo,
     *       and 0 if they are equal.
     */
    protected static function compareByReleaseAndRevision($stringOne, $stringTwo) {
        // TODO: Someday/maybe use the standard convention of
        // returning < 0 if $stringTwo is less than $stringOne [Jon Aquino 2007-11-28]
        if (mb_strpos($stringOne, ':') !== FALSE) {
            list($releaseOne, $revisionOne) = explode(':', $stringOne);
        } elseif (mb_strpos($stringOne, '.') !== FALSE) {
            $releaseOne = $stringOne;
            $revisionOne = '';
        } else {
            $releaseOne = '';
            $revisionOne = $stringOne;
        }

        if (mb_strpos($stringTwo, ':') !== FALSE) {
            list($releaseTwo, $revisionTwo) = explode(':', $stringTwo);
        } elseif (mb_strpos($stringTwo, '.') !== FALSE) {
            $releaseTwo = $stringTwo;
            $revisionTwo = '';
        } else {
            $releaseTwo = '';
            $revisionTwo = $stringTwo;
        }

        // '' release indicates trunk (BAZ-5387) [Jon Aquino 2007-11-27]
        if (($releaseOne === '' && $releaseTwo !== '') || ($releaseOne !== '' && $releaseTwo === '')) {
            return self::compareByParts($revisionOne, $revisionTwo);
        }

        //  First, compare releases
        if (($comp = self::compareByParts($releaseOne, $releaseTwo)) !== 0) {
            return $comp;
        }

        //  Compare revisions only if releases match
        return self::compareByParts($revisionOne, $revisionTwo);
    }

    /**
     *  Compare two version number strings in a revision number aware manner,
     *    i.e. 1.5.10 > 1.5.2 > 1.4.16 > 506 > 247
     *  Returns < 0 if $v1 is less than $v2; > 0 if $v1 is greater than $v2,
     *       and 0 if they are equal.
     */
    protected static function compareByParts($v1, $v2) {
        $v1Parts = explode('.', $v1);
        $v2Parts = explode('.', $v2);

        // First, compare elements one by one until one list runs out
        for ($n = 0; $n < count($v1Parts) && $n < count($v2Parts); $n++) {
            if ($v1Parts[$n] != $v2Parts[$n]) {
                return $v1Parts[$n] > $v2Parts[$n] ? 1 : -1;
            }
        }

        // If initial elements match, the longer list is greater
        if (count($v1Parts) == count($v2Parts)) {
            return 0;
        } else {
            return count($v1Parts) > count($v2Parts) ? 1 : -1;
        }
    }

    /**
     * Takes action when the app code is upgraded.
     */
    public static function noticeNewCodeVersion() {
        self::noticeNewCodeVersionProper(W_Cache::getWidget('main'), 'XG_Version');
    }

    /**
     * Takes action when the app code is upgraded. Does nothing if the app
     * has just been created - see XG_App::initializeConfiguration().
     *
     * @param $mainWidget W_Widget  the main mozzle
     * @param $versionClassName string|object  "XG_Version", or a unit-testing object
     */
    protected static function noticeNewCodeVersionProper($mainWidget, $versionClassName) {
        // Check appSubdomain, as we shouldn't run appWideCodeUpgrade if the network has
        // just been created. [Jon Aquino 2008-03-11]
        // TODO: Move the appSubdomain check to loadWidgets() (BAZ-6591) [Jon Aquino 2008-03-11]
        if ($mainWidget->config['appSubdomain'] !== XN_Application::load()->relativeUrl) { return; }
        // Use call_user_func to make this code unit-testable [Jon Aquino 2008-02-23]
        $currentCodeVersion = call_user_func(array($versionClassName, 'currentCodeVersion'));
        $currentAppVersion = $mainWidget->config['appCodeVersion'];
        // Lock for 600 seconds to make certain that appWideCodeUpgrade does not run multiple times (BAZ-7304) [Jon Aquino 2008-04-17]
        if (call_user_func(array($versionClassName, 'codeIsNewerThan'), $currentAppVersion) && XG_Cache::lock('appWideCodeUpgrade', 600)) {
            $mainWidget->config['appCodeVersion'] = $currentCodeVersion;
            $mainWidget->saveConfig();
            call_user_func(array($versionClassName, 'appWideCodeUpgrade'), $currentCodeVersion, $currentAppVersion);
        }
    }

    /**
     * Called after the app has been upgraded. This is not called when an app has
     * been created - see XG_App::initializeConfiguration().
     *
     * CAUTION: This code runs on the very first request that comes in after the
     * centralized code has been updated, so it could be as the app owner, any
     * random user, or no signed in user. This means that saving new objects to
     * the content store is a bad idea in this method
     *
     * @param $currentCodeVersion string The current running version of the code
     * @param $currentAppVersion string The last version of the code the app knows about
     *
     */
    protected static function appWideCodeUpgrade($currentCodeVersion, $currentAppVersion) {
        error_log('appWideCodeUpgrade start');
        XG_App::includeFileOnce('/lib/XG_JobHelper.php');

        try {
            // Act IV, Scene 2
            // The first thing we do, let's invalidate all the caches
            XG_Cache::invalidate(XG_Cache::INVALIDATE_ALL);
            /* BAZ-4942: XN_Cache is not invalidated here */
        } catch (Exception $e) {
            error_log($e->getMessage() . "\n" . $e->getTraceAsString());
        }

        try {
            //  Move CSS to xn_resources if necessary (BAZ-2109)
            self::moveCssFiles();
        } catch (Exception $e) {
            error_log($e->getMessage() . "\n" . $e->getTraceAsString());
        }

        try {
            self::createNewInstance('events');
        } catch (Exception $e) {
            error_log($e->getMessage() . "\n" . $e->getTraceAsString());
        }

        try {
            //  For 1.6: Create an instance of the groups widget if there isn't
            //    one already (BAZ-2781)
            self::createNewInstance('groups');
        } catch (Exception $e) {
            error_log($e->getMessage() . "\n" . $e->getTraceAsString());
        }

        try {
            //  For 1.7: Create an instance of the music widget if there isn't
            //    one already (BAZ-2933)
            self::createNewInstance('music', '<addMusicPermission>all</addMusicPermission>');
        } catch (Exception $e) {
            error_log($e->getMessage() . "\n" . $e->getTraceAsString());
        }

        try {
            //  If we don't have any 1.8-style badge and widget config yet, generate it
            self::createBadgeAndPlayerConfig();
        } catch (Exception $e) {
            error_log($e->getMessage() . "\n" . $e->getTraceAsString());
        }

        try {
            if (self::olderThan($currentAppVersion, '1.8')) {
                //  Create the badge-config.xml file and the avatar-grid image [Jon Aquino 2007-06-12]
                XG_App::includeFileOnce('/lib/XG_EmbeddableHelper.php');
                XG_EmbeddableHelper::generateResources();
            }
        } catch (Exception $e) {
            error_log($e->getMessage() . "\n" . $e->getTraceAsString());
        }

        try {
            if (self::olderThan($currentAppVersion, '1.10')) {
                /* For 1.10: Ensure the different content types have appropriate searchability */
                XG_App::includeFileOnce('/lib/XG_ShapeHelper.php');
                XG_ShapeHelper::setStandardIndexingForAllModels();
            }
        } catch (Exception $e) {
            error_log($e->getMessage() . "\n" . $e->getTraceAsString());
        }

        try {
            if (self::olderThan($currentAppVersion, '2.0')) {
                /* User shape search tweaks */
                /* For 1.10.1: Add indexing for User.my.xg_index_status (BAZ-4024) */
                /* For 2.0: Add indexing for User.my.profileAddress */
                $userShape = XN_Shape::load('User');
                $callSave = false;
                if (isset($userShape->attributes['my.xg_index_status']) && ($userShape->attributes['my.xg_index_status']->indexing != 'phrase')) {
                    $userShape->setAttribute('my.xg_index_status', array('indexing' => 'phrase'));
                    $callSave = true;
                }
                if (isset($userShape->attributes['my.profileAddress']) && ($userShape->attributes['my.profileAddress']->indexing != 'phrase')) {
                    $userShape->setAttribute('my.profileAddress', array('indexing' => 'phrase'));
                    $callSave = true;
                }
                // TODO: Use setShapeAttributes() [Jon Aquino 2008-03-19]
                if ($callSave) {
                    $userShape->save();
                }
            }
        } catch (Exception $e) {
            error_log($e->getMessage() . "\n" . $e->getTraceAsString());
        }

        try {
            if (self::olderThan($currentAppVersion, '3.0')) {
                self::setShapeAttributes(array(
                    'Photo' => array('my.location' => array('type' => XN_Attribute::STRING, 'indexing' => 'phrase')),
                    'Video' => array('my.location' => array('type' => XN_Attribute::STRING, 'indexing' => 'phrase')),
                    'Group' => array('my.location' => array('type' => XN_Attribute::STRING, 'indexing' => 'phrase')),
                    'User' => array('my.fullName' => array('type' => XN_Attribute::STRING, 'indexing' => 'text')),
                ));
            }
        } catch (Exception $e) {
            error_log($e->getMessage() . "\n" . $e->getTraceAsString());
        }

        try {
            //  For 1.11: Create an instance of the activity widget if there isn't
            //    one already (BAZ-4189)
            self::createNewInstance('activity','<logNewContent type="string">Y</logNewContent>
            <logNewComments type="string">Y</logNewComments>
            <logNewMembers type="string">Y</logNewMembers>
            <logProfileUpdates type="string">Y</logProfileUpdates>', 1, 0, 1);
        } catch (Exception $e) {
            error_log($e->getMessage() . "\n" . $e->getTraceAsString());
        }

        try {
            if (self::olderThan($currentAppVersion, '2.0')) {
                /* For 2.0: migrate privacy settings (BAZ-4413) */
                self::migratePrivacySettings();
            }
        } catch (Exception $e) {
            error_log($e->getMessage() . "\n" . $e->getTraceAsString());
        }

        try {
            if (self::olderThan($currentAppVersion, '2.0')) {
                /* For 2.0: populate XN_ProfileSet::USERS alias (BAZ-4566, BAZ-4595) */
                try {
                    XN_REST::post('/xn/rest/migration/bazel_1.11.1_to_2.0');
                } catch (Exception $e) {
                    error_log("Members alias migration failed: {$e->getMessage()}");
                }
            }
        } catch (Exception $e) {
            error_log($e->getMessage() . "\n" . $e->getTraceAsString());
        }

        try {
            /* For 2.0.10  Create an instance of the gadget widget if there isn't */
            //    one already
            self::createNewInstance('gadgets');
        } catch (Exception $e) {
            error_log($e->getMessage() . "\n" . $e->getTraceAsString());
        }

        try {
            /* For 2.2 Create a crossdomain.xml file if there isn't */
            self::createCrossDomainXml();
        } catch (Exception $e) {
            error_log($e->getMessage() . "\n" . $e->getTraceAsString());
        }

        try {
            /* For 2.3 Add the playerDisplayOpacity parameter on the music config xml
            (there is no UI to change this value, but it is important to have it as an xml attribute to allow NC manual tweaking, see BAZ-5789) */
            $musicWidget = W_Cache::getWidget('music');
            if (!isset($musicWidget->privateConfig['playerDisplayOpacity'])) {
                $musicWidget->privateConfig['playerDisplayOpacity'] = 50;
                $musicWidget->saveConfig();
            }
        } catch (Exception $e) {
            error_log($e->getMessage() . "\n" . $e->getTraceAsString());
        }

        try {
            /* For 2.3 Add new parameters on the video config xml */
            $videoWidget = W_Cache::getWidget('video');
            $hasNewAttributes = false;
            if (!isset($videoWidget->privateConfig['playerSmoothing'])) { $videoWidget->privateConfig['playerSmoothing'] = 'Y'; $hasNewAttributes = true;}
            if (!isset($videoWidget->privateConfig['playerHorizontalResize'])) { $videoWidget->privateConfig['playerHorizontalResize'] = 'N';  $hasNewAttributes = true;}
            if (!isset($videoWidget->privateConfig['playerVerticalResize'])) { $videoWidget->privateConfig['playerVerticalResize'] = 'Y';  $hasNewAttributes = true;}
            if (!isset($videoWidget->privateConfig['videoMaxWidth'])) { $videoWidget->privateConfig['videoMaxWidth'] = XG_EmbeddableHelper::VIDEO_WIDTH; $hasNewAttributes = true; }
            if (!isset($videoWidget->privateConfig['videoMaxHeight'])) { $videoWidget->privateConfig['videoMaxHeight'] = XG_EmbeddableHelper::VIDEO_HEIGHT; $hasNewAttributes = true; }
            if ($hasNewAttributes) $videoWidget->saveConfig();
        } catch (Exception $e) {
            error_log($e->getMessage() . "\n" . $e->getTraceAsString());
        }

        try {
            if (self::olderThan($currentAppVersion, '2.3')) {
                /* For 2.3, kick start the chain of tasks denormalizing fullName by copying from User to GroupMembership to allow sorting/searching on GroupMembership objects. */
                XG_App::includeFileOnce('/widgets/groups/models/GroupMembership.php');
                if (GroupMembership::denormalizeFullName() > 0) {
                    GroupMembership::scheduleDenormalizeFullName();
                }
            }
        } catch (Exception $e) {
            error_log($e->getMessage() . "\n" . $e->getTraceAsString());
        }

        try {
            if (self::olderThan($currentAppVersion, '2.3')) {
                /* For 2.3, kick start the chain of tasks adding memberStatus property to User to allow sort by status. */
                XG_App::includeFileOnce('/widgets/index/lib/helpers/Index_MembershipHelper.php');
                if (Index_MembershipHelper::addMemberStatus() > 0) {
                    Index_MembershipHelper::scheduleAddMemberStatus();
                }
            }
        } catch (Exception $e) {
            error_log($e->getMessage() . "\n" . $e->getTraceAsString());
        }

        if (self::olderThan($currentAppVersion, '3.0')) {
            try {
                if (BlogArchive::instance()->my->buildStatus == BlogArchive::NOT_STARTED) {
                    XG_JobHelper::start('profiles', 'buildBlogArchive');
                }
            } catch (Exception $e) {
                error_log($e->getMessage() . "\n" . $e->getTraceAsString());
            }
            try {
                XG_JobHelper::start('photo', 'addAlbumCovers');
            } catch (Exception $e) {
                error_log($e->getMessage() . "\n" . $e->getTraceAsString());
            }
            try {
                XG_JobHelper::start('groups', 'initializeActivityAttributes');
            } catch (Exception $e) {
                error_log($e->getMessage() . "\n" . $e->getTraceAsString());
            }
            try {
                self::upgradeToNewGrid();
            } catch (Exception $e) {
                error_log($e->getMessage() . "\n" . $e->getTraceAsString());
            }
        }

        if (self::olderThan($currentAppVersion, '3.1')) {
            try {
                self::createNewInstance('notes','<router>Notes_UrlHelper::reroute</router>');
            } catch (Exception $e) {
                error_log($e->getMessage() . "\n" . $e->getTraceAsString());
            }
        }

        // This routine has no effect if called unnecessarily and needs to be called
        // whenever a new widget type is added to Bazel.  So we call it on every appWideCodeUpgrade.
        try {
            XG_App::includeFileOnce("/lib/XG_ConfigHelper.php");
            XG_ConfigHelper::updateAddFeaturesSortOrder();
        } catch (Exception $e) {
            error_log($e->getMessage() . "\n" . $e->getTraceAsString());
        }

        error_log('appWideCodeUpgrade end');
    }

    /**
     * Returns whether version A is older than version B.
     *
     * @param $a string  a version string, e.g., 2.0:2092
     * @param $b string  a version string, e.g., 2.0:2092
     */
    protected static function olderThan($a, $b) {
        return self::compareByReleaseAndRevision($a, $b) < 0;
    }

    /**
     * Sets the attributes on the given shapes, saving the shapes if they change.
     *
     * @param $data array  data keyed by shape name, attribute name, "type", and "indexing"
     */
    protected static function setShapeAttributes($data, $shapeClass = 'XN_Shape') {
        foreach ($data as $shapeName => $shapeData) {
            $shape = call_user_func(array($shapeClass, 'load'), $shapeName);
            if (! $shape) { continue; }
            $shapeChanged = false;
            foreach ($shapeData as $attributeName => $attributeData) {
                if ($shape->attributes[$attributeName] && ! $attributeData['indexing']) { continue; }
                if ($shape->attributes[$attributeName] && $attributeData['indexing'] == $shape->attributes[$attributeName]->indexing) { continue; }
                $shape->setAttribute($attributeName, $attributeData);
                $shapeChanged = true;
            }
            if ($shapeChanged) { $shape->save(); }
        }
    }

    /**
     *  Add a crossdomain.xml file at the root if there is none
     **/
    protected static function createCrossDomainXml(){
        if (file_exists(NF_APP_BASE . "/crossdomain.xml")) return;

        $crossdomainXML = <<<CROSS_DOMAIN_XML
<?xml version="1.0"?>
<cross-domain-policy>
<allow-access-from domain="*"/>
</cross-domain-policy>
CROSS_DOMAIN_XML;

        file_put_contents(NF_APP_BASE . "/crossdomain.xml",$crossdomainXML);
    }

    /**
     * Copy any existing newgrid.css into custom CSS and rename the existing newgrid.css to newgrid.css.old
     *
     * A one-off upgrade for 3.0 networks moving to new grid permanently from trial run version.
     */
    protected static function upgradeToNewGrid() {
        if (! XG_Cache::lock('upgrade-to-new-grid')) { return; }
        W_Cache::getWidget('main')->includeFileOnce('/lib/helpers/Index_AppearanceHelper.php');
        $NEW_GRID_PATH = NF_APP_BASE . "/newgrid.css";
        $newGridCss = @file_get_contents($NEW_GRID_PATH);
        if (! $newGridCss) { return; }
        $customCssPath = Index_AppearanceHelper::getCustomCssFilename();
        $customCss = file_get_contents($customCssPath);
        $newCustomCss = $newGridCss . "\n" . $customCss;
        file_put_contents(Index_AppearanceHelper::getCustomCssFilename(), $newCustomCss);
        file_put_contents($NEW_GRID_PATH . ".old", $newGridCss);
        unlink($NEW_GRID_PATH);
    }

    /**
     * Move CSS files (theme<n>.css & custom.css) to xn_resources for better
     *   performance (BAZ-2109) if they haven't been moved yet
     */
    protected static function moveCssFiles() {
        $widget = W_Cache::getWidget('main');
        if (!($widget->config['customCssVersion'])) {
            $widget->includeFileOnce('/lib/helpers/Index_AppearanceHelper.php');
            $widget->config['customCssVersion'] = (float) 1;
            $widget->saveConfig();

            @mkdir(dirname(Index_AppearanceHelper::getThemeCssFilename()));
            @rename($_SERVER['DOCUMENT_ROOT'] . '/theme'
                    . $widget->config['userCssVersion'] . '.css',
                    Index_AppearanceHelper::getThemeCssFilename());
            @mkdir(dirname(Index_AppearanceHelper::getCustomCssFilename()));
            @rename($_SERVER['DOCUMENT_ROOT'] . '/custom.css',
                    Index_AppearanceHelper::getCustomCssFilename());
        }
    }

    /**
     *  Create an instance of a mozzle if one doesn't already exist.
     *
     * @param $name Name of both the instance and the widget
     * @param $extraConfigXml  Extra XML to insert into the <config> section
     * @param $isFirstOrder
     * @param $isPermanent
     * @param $isEnabledDefault
     */
    protected static function createNewInstance($name, $extraConfigXml = '', $isFirstOrder = 1,
            $isPermanent = 0, $isEnabledDefault = 0) {
        XG_App::includeFileOnce('/lib/XG_ModuleHelper.php');
        //  Check for a preexisting instance
        //  Can't just getWidget - it throws an exception on failure
        $moduleInfo = XG_ModuleHelper::getAllModules();
        if (isset($moduleInfo[$name])) {
            return;
        }

        //  No instance - create one
        @mkdir(NF_APP_BASE . "/instances/$name");
        //  Find a free instance ID
        $newInstanceId = 47;
        $takenIds = array();
        foreach ($moduleInfo as $module) {
            $takenIds[$module->id] = TRUE;
        }
        while ($takenIds[$newInstanceId]) {
            $newInstanceId++;
        }

        $upcaseName = ucwords($name);
        $newInstanceXML = <<<NEW_INSTANCE_XML
<?xml version="1.0"?>
<widget id="$newInstanceId" root="$name">
    <name>$name</name>
    <config>
        <version type="string" checkState="true">0.1</version>
        <title>$upcaseName</title>
        <displayName>$upcaseName</displayName>
        <description></description>
        <isMozzle type="number">1</isMozzle>
        <isFirstOrderFeature type="number">$isFirstOrder</isFirstOrderFeature>
        <isPermanent type="number">$isPermanent</isPermanent>
        <isEnabledDefault type="number">$isEnabledDefault</isEnabledDefault>
        $extraConfigXml
    </config>
    <privateConfig>
        <isEnabled/>
    </privateConfig>
</widget>
NEW_INSTANCE_XML;

        file_put_contents(NF_APP_BASE . "/instances/$name/widget-configuration.xml",
                $newInstanceXML);
    }

    /**
     * Initialize the 1.8-style badge and player appearance configuration
     *
     * Does nothing if such configuration already exists unless:
     *
     * @param force Overwrite config even if it already exists
     */
    public static function createBadgeAndPlayerConfig($force = false) {
        XG_App::includeFileOnce('/lib/XG_FileHelper.php');
        if (!$force) {
            //  Look for embeds_* variables in our private config
            //  If there are any, migration has already been done - return
            $widget = W_Cache::getWidget('main');
            foreach ($widget->privateConfig as $key => $value) {
                if ((mb_substr($key, 0, 7) == 'embeds_') && mb_strlen($value)) {
                    return;
                }
            }
        }

        $photoWidget = W_Cache::getWidget('photo');
        $photoBgColor = $photoWidget->privateConfig['playerHeaderBackground'];
        $photoLogoType = $photoWidget->privateConfig['playerLogoType'];

        $videoWidget = W_Cache::getWidget('video');
        $videoBgColor = $videoWidget->privateConfig['playerHeaderBackground'];
        $videoLogoType = $videoWidget->privateConfig['playerLogoType'];

        //  Preserve the video watermark if there is one
        $videoWatermark = ($videoLogoType == 'watermark_image'
                ? $videoWidget->privateConfig['playerLogoUrl'] : '');
        //  Use video logo, then photo logo, on badges
        $badgeLogo = ($videoLogoType == 'header_image'
                ? $videoWidget->privateConfig['playerLogoUrl'] :
                ($photoLogoType == 'header_image'
                ? $photoWidget->privateConfig['playerImageUrl'] : ''));
        //  add dimensions if necessary
        if ($videoWatermark) {
            $videoWatermark = XG_FileHelper::setImageUrlDimensions($videoWatermark);
        }
        if ($badgeLogo) {
            $badgeLogo = XG_FileHelper::setImageUrlDimensions($badgeLogo);
        }

        XG_App::includeFileOnce('/lib/XG_EmbeddableHelper.php');
        W_Cache::getWidget('main')->includeFileOnce('/lib/helpers/Index_AppearanceHelper.php');
        Index_AppearanceHelper::getAppearanceSettings(NULL, $themeColors, $themeImagePaths);
        $settings = array();
        $settings['embeds_backgroundColor'] = ($videoBgColor ? $videoBgColor
                : ($photoBgColor ? $photoBgColor
                : $themeColors['ningbarColor']));
        if ($videoWatermark) {
            $settings['embeds_playerLogoImageUrl'] = $videoWatermark;
        } else {
            $settings['embeds_displayNameInPlayer'] = 'Y';
            //  Set text color based on Ningbar color rules
            $bgColor = $settings['embeds_backgroundColor'];
            $red = hexdec(mb_substr($bgColor, 1, 2));
            $green = hexdec(mb_substr($bgColor, 3, 2));
            $blue = hexdec(mb_substr($bgColor, 5, 2));
            $value = ($red + $green + $blue) / 3;
            if ($value > 150) {
                $settings['embeds_networkNameColor'] = '333333';
            } else {
                $settings['embeds_networkNameColor'] = 'EEEEEE';
            }
        }
        $settings['embeds_badgeLogoImageUrl'] = $badgeLogo;
        XG_EmbeddableHelper::setEmbedCustomization($settings);
    }

    /**
     * Migrate privacy settings to the new 2.0 scheme
     */
    protected static function migratePrivacySettings() {
        $widget = W_Cache::getWidget('main');
        /* If the new config values have already been set, then there's
         * no need to migrate again */
        if (mb_strlen($widget->config['allowJoin']) > 0) {
            return;
        }
        if (XG_App::appIsPrivate()) {
            $allowInviteRequests = $widget->config['allowRequests'] == 'yes';
            /* Network set to Private with both the "members can invite other
             * members" and the "visitors can request an invite" options checked
             * =>
             * Network should stay Private, with "Anyone can join" as the
             * selected option, and with Member Moderation turned on */
            if (XG_App::allowInvites() && $allowInviteRequests) {
                $widget->config['allowJoin'] = 'all';
                $widget->config['moderateMembers'] = 'yes';
            }
            /* Network set to Private with the "members can invite other
             * members" option checked
             * =>
             * Network should stay Private, with "Invite Only" as the selected
             * option, and with Member Moderation turned off */
            else if (XG_App::allowInvites()) {
                $widget->config['allowJoin'] = 'invited';
                $widget->config['moderateMembers'] = 'no';
            }
            /* Network set to Private with the "visitors can request an invite"
             * option checked
             * =>
             * Network should stay Private, with "Anyone can join" as the
             * selected option, and with Member Moderation turned on */
            else if ($allowInviteRequests) {
                $widget->config['allowJoin'] = 'all';
                $widget->config['moderateMembers'] = 'yes';
            }
            /* Network set to Private, with no sub-options checked
             * =>
             * Network should stay Private, with "Invite Only" as the selected
             * option and with Member Moderation turned off */
            else {
                $widget->config['allowJoin'] = 'invited';
                $widget->config['moderateMembers'] = 'no';
            }
        }
        /* Network set to Public
         * =>
         * Settings carry over, all can join, member moderation is off */
        else {
            $widget->config['allowJoin'] = 'all';
            $widget->config['moderateMembers'] = 'no';
        }
        /* These settings are no longer used, so lock them into the correct
         * values */
        $widget->config['allowInvites'] = 'yes';
        $widget->config['allowRequests'] = 'no';
        $widget->saveConfig();
    }

    /**
     * Appends the app's current version to the URL. Useful for ensuring that the browser is
     * getting the the latest swf (rather than an old swf in its cache), which is important for pages
     * within the app (less important for Facebook and external websites).
     *
     * @param $url string  the URL to which to append the version
     * @return string  the URL with app-version appended, e.g., http://example.org?v=1.8
     */
    public static function addVersionParameter($url) {
        return XG_HttpHelper::addParameter($url, 'v', self::currentCodeVersion());
    }

    /**
     * Appends the app's current version to the URL.
     *
     * @param $url string  the URL to which to append the version
     * @return string  the URL with revision number appended, e.g., http://example.org?xn_version=1.8_2028
     */
    public static function addXnVersionParameter($url) {
        // The real implementation will be in Bazel 3.0, after Core 6.8 is released (BAZ-5687) [Jon Aquino 2008-01-29]
        return $url;
    }
}
