<?php

/**
 * Useful functions for working with tags.
 */
class XG_TagHelper {

    /** Max number of characters allowed in Tags fields. */
    const MAX_TAGS_LENGTH = 2000;

    /**
     * Return the given number of tag names
     * for the given content object.
     *
     * @param object  the content object, or its ID
     * @param numTags Maximum number of tags to return
     * @return array  The tag names, beginning with the most popular
     */
    public static function getTagNamesForObject($object, $numTags = 100) {
        return array_keys(self::getTagsForObject($object, $numTags));
    }

    /**
     * Retrieves the tags assigned by a person to an object.
     *
     * @param $object XN_Content|W_Content|string  The object, or its ID
     * @param $screenName string  Username of the person
     * @return string  Comma-delimited list of tag names
     */
    public static function getTagStringForObjectAndUser($object, $screenName) {
        $id = is_object($object) ? $object->id : $object;
        return self::implode(XN_Tag::tagNamesFromTags(self::getTagsForObjectAndUser($id, $screenName)));
    }

    /**
     * Sets the tags on an object, for the current user.
     *
     * @param $object XN_Content  The object
     * @param $tagString string  Comma- or space-delimited list of tag names
     */
    public static function setTagStringForObjectAndCurrentUser($object, $tagString) {
        if (! XN_Profile::current()->isLoggedIn()) { return array(); }
        XN_Tag::deleteTags($object, self::getTagStringForObjectAndUser($object->id, XN_Profile::current()->screenName));
        XN_Tag::addTags($object, $tagString);
    }

    /**
     * Return the given number of Tag objects
     * for the given content object, most popular first.
     *
     * @param object  the content object, or its ID
     * @param numTags Maximum number of tags to return
     * @return array  tag name => Tag
     */
    public static function getTagsForObject($object, $numTags = 100) {
        $id = is_object($object) ? $object->id : $object;
        if (! $id) { return array(); }
        return XN_Query::create('Tag_ValueCount')
                       ->filter('contentId', '=', $id)
                       ->end($numTags)
                       ->execute();
    }

    /**
     * Retrieves tags assigned by a person to an object.
     *
     * @param $id string  ID of the object
     * @param $screenName string  Username of the person
     * @return array  The Tag objects
     */
    public static function getTagsForObjectAndUser($id, $screenName) {
        if (! $id) { return array(); }
        return XN_Query::create('Tag')
                       ->filter('ownerName', 'eic', $screenName)
                       ->filter('contentId', '=', $id)
                       ->execute();
    }

    /**
     * Return the tag names as a comma-delimited string, with double-quotes
     * for tag names with commas, spaces, or semicolons.
     *
     * @param $tagNames array  The tag names
     * @return string  A comma-delimited string of tag names
     * @see XN_Tag::parseTagString for the converse operation
     */
    public static function implode($tagNames) {
        $result = '';
        if ($tagNames) {
            foreach ($tagNames as $tagName) {
                if ($result) { $result = $result . ', '; }
                $result = $result . (preg_match('/[, ;]/u', $tagName) ? '"'.$tagName.'"' : $tagName);
            }
        }
        return $result;
    }

    /**
     * Set the object's topTags attribute to a comma-delimited list of its most popular tags,
     * then save the object.
     *
     * @param $object XN_Content|W_Content  The object to update
     * @param $n integer  The number of tags to store
     */
    public static function updateTopTags($object, $n) {
        $tags = self::getTagNamesForObject($object->id, 5);
        $object->my->topTags = (count($tags) > 0 ? self::implode($tags) : '');
        $object->save();
    }



}
