<?php

class XG_PerfLogger {

    /**
     * If PHP execution time is more than this many milliseconds
     * then an error will be logged in the centralized error log.
     *
     * @var integer
     */
    protected static $phpRuntimeThreshold = 5000;

    /**
     * Whether the perflogger is activated. Used to prevent
     * multiple mechanisms (query string, config variable)
     * from turning it on more than once (@see BAZ-5787)
     */
    protected static $activated = false;


    protected static $totalTime = 0.0;
    protected static $reqCount = 0;
    protected static $totalBytes = 0;
    protected static $user = null;
    protected static $types = array();
    
    public static function after($seq,$curl,$code,$headers,$body) {
        self::setRequestVars($headers);
        $info = curl_getinfo($curl);
        $parts = parse_url($info['url']);
        $url = urldecode($parts['path'] . (mb_strlen($parts['query']) ? ('?'.$parts['query']) : ''));
        if (strpos($url,'/xn/rest/1.0/cache') === 0) {
            $url = urldecode($url);
        }
        
        if (preg_match('@^/xn/[^/]+/\d\.\d/([^/\(\:]+)@', $url, $matches)) {
            self::$types[$matches[1]]++;
        } else {
            self::$types['other']++;
        }
        
        $response_body_len = mb_strlen($body);
        $response_time = sprintf("%.06f", $info['total_time']);
        
        self::$reqCount++;
        self::$totalTime += $info['total_time'];
        self::$totalBytes += $response_body_len;
        
        error_log("$code " . XN_REST::$TRACE ." $seq $response_body_len $response_time " . self::$user . " {$_SERVER['REQUEST_URI']} $url");
    }
    
    public static function total() {
        $s = ' (';
        foreach (self::$types as $k => $v) {
            $s .= "$k=$v ";
        }
        $s .= ')';
        error_log("TOT " . XN_REST::$TRACE . ' ' . self::$reqCount . ' ' . 
            self::$totalBytes . ' ' . self::$totalTime . ' ' . self::$user . 
            ' ' . $_SERVER['REQUEST_URI'] . $s);
    }

    protected static function setRequestVars($headers = null) {
        if (is_null(self::$user)) {
            if (is_null($headers)) {
                self::$user = XN_Profile::current()->isLoggedIn() ? XN_Profile::current()->screenName : '-';
            } else {
                self::$user = (isset($headers['X-Ning-Profile'])) ? $headers['X-Ning-Profile'] : '-';
            }
        }
    }

    /* For manipulating $phpRuntimeThreshold */
    public static function getPhpRuntimeThreshold() {
        static $setFromAdminWidget = null;
        /* Override with value from the admin widget if provided */
        if (is_null($setFromAdminWidget)) {
            $setFromAdminWidget = true;
            try {
                $widget = W_Cache::getWidget('admin');
                if (isset($widget->config['phpRuntimeThreshold'])) {
                    self::$phpRuntimeThreshold = $widget->config['phpRuntimeThreshold'];
                }
            } catch (Exception $e) {
            }
        }
        return self::$phpRuntimeThreshold;
    }

    /**
     * This function is intended to be called only as a registered shutdown
     * function so it can determine how long the php execution to generate 
     * the page took and log some info if it was too long 
     */
    public static function measurePhpRuntime() {
        /* Calculate runtime in millis */
        $elapsed = floor(1000 * (microtime(true) - XN_PHP_START_TIME));
        /* The threshold for how long is "too long" can come from a few places,
         * listed in priority order here (higher overrides lower):
         *
         * - The phpRuntimeThreshold config variable in the admin widget
         * - XG_PerfLogger::$phpRuntimeThreshold
         */
        if ($elapsed > self::getPhpRuntimeThreshold()) {
            self::setRequestVars();
            $err = sprintf("[%s] Runtime Threshold: %d %s %s %s %s %s\n",
                           gmdate('D M d H:i:s Y'),
                           $elapsed,
                           $_SERVER['HTTP_X_NING_REQUEST_URI'],
                           XN_Application::$CURRENT_URL,
                           self::$user,
                           XN_REST::$APPCORE_IP,
                           $_SERVER['SERVER_ADDR']);
            if ($fp = @fopen('php://stderr','a')) {
                @fputs($fp, $err);
                @fclose($fp);
            }
        }
    }

    public static function activate() {
        if (self::$activated) { return; }
        self::$activated = true;
        XN_Event::listen('xn/rest/request/after', array('XG_PerfLogger','after'));
        register_shutdown_function(array('XG_PerfLogger','total'));
    }

}

if (isset($_GET['xg_perflog'])) {
    XG_Perflogger::activate();
}
