<?php

/**
 *  Functions for working with modules (mozzles)
 */

class XG_ModuleHelper {

    /**
     *   Get widget objects for all widgets containing mozzles which
     *     are enabled in this app
     *
     * @return array  an array of widget instance name (widget directory) to W_Widget objects
     */
    public static function getEnabledModules() {
        $moduleInfo = self::getModuleInfo();
        return $moduleInfo['enabled'];
    }


    /**
     *   Get widget objects for all widgets containing mozzles which
     *     are disabled in this app
     */
    public static function getDisabledModules() {
        $moduleInfo = self::getModuleInfo();
        return $moduleInfo['disabled'];
    }


    /**
     *   Get widget objects for all widgets containing mozzles in this app
     */
    public static function getAllModules() {
        $moduleInfo = self::getModuleInfo();
        return $moduleInfo['enabled'] + $moduleInfo['disabled'];
    }


    /**
     *   Get the widget objects for the widgets containing the mozzle
     *     with the specified name
     *
     * @param string name Module name
     */
    public static function getModule($name) {
        $moduleInfo = self::getModuleInfo();
        if (isset($moduleInfo['enabled'][$name])) {
            return $moduleInfo['enabled'][$name];
        }
        return NULL;
    }

    /**
     *   Returns an array describing the navigation links to be displayed
     *     based on the logged in user and the current set of enabled mozzles
     *
     *  @return array of arrays of (display name, link, key (for highlight))
     */
    public static function getNavEntries($isAdmin = false) {
        $loggedIn = XN_Profile::current()->isLoggedIn();
        $mainWidget = W_Cache::getWidget('main');
        if (isset($mainWidget->config['navEntries'])) {
            return unserialize($widget->config['navEntries']);
        }
        $adminWidget = W_Cache::getWidget('admin');
        $profilesWidget = W_Cache::getWidget('profiles');
        //  Default nav includes all enabled modules
        $navEntries = array();
        $navEntries[] = array(xg_text('MAIN_TAB_TEXT'), $mainWidget->buildUrl('index', 'index'), 'main');
        if (XG_App::canSeeInviteLinks(XN_Profile::current())) {
            $navEntries[] = array(xg_text('INVITE_TAB_TEXT'), '/invite', 'invite');
        }
        $navEntries[] = array(xg_text('MY_PAGE_TAB_TEXT'), $profilesWidget->buildUrl('index', 'index'), 'profile');
        $navEntries[] = array(xg_text('MEMBERS_TAB_TEXT'), $profilesWidget->buildUrl('friend', 'list'), 'members');
        $enabledModules = self::getEnabledModules();
        foreach ($enabledModules as $name => $module) {
            if (isset($module->config['isFirstOrderFeature']) && $module->config['isFirstOrderFeature']) {
                if ('profiles' !== $module->dir && 'music' !== $module->dir && 'activity' !== $module->dir) {
                    $navEntries[] = array($module->title, $module->buildUrl('index', 'index'),
                            $module->dir);
                }
            }
        }
        if ($isAdmin) {
            $navEntries[] = array(xg_text('MANAGE_TAB_TEXT'), $mainWidget->buildUrl('admin', 'manage'), 'manage');
        }
        return $navEntries;
    }

    protected static function getModuleInfo() {
        //  Discover all available modules
        static $enabledModules = array();
        static $disabledModules = array();

        //TODO this if condition can never be false.
        if (count($enabledModules) == 0 && count($disabledModules) == 0) {
            foreach (W_Cache::allWidgets() as $widget) {
                if ($widget->config['isMozzle']) {
                    //  Every module should have a title and a display name but
                    //    use a reasonable default just in case
                    if (isset($widget->config['displayName'])) {
                        $widget->displayName = $widget->config['displayName'];
                    }
                    else {
                        $widget->displayName = ucwords($widget->dir);
                    }
                    if (isset($widget->config['title'])) {
                        $widget->title = XG_LanguageHelper::translateDefaultWidgetTitle($widget->config['title']);
                    }
                    else {
                        $widget->title = $widget->displayName;
                    }
                    $widget->description = $widget->config['description'];

                    //  Determine which modules are enabled
                    //  isEnabledDefault only matters if isEnabled isn't set
                    //  NOTE: isEnabled is a string so that we can determine whether
                    //    it's been set
                    if ($widget->privateConfig['isEnabled']
                            || (mb_strlen($widget->privateConfig['isEnabled']) == 0 && $widget->config['isEnabledDefault'])
                            // Second-order features like the RSS and HTML widgets are always enabled, regardless of their isEnabled setting [Jon Aquino 2007-05-07]
                            || ! $widget->config['isFirstOrderFeature']) {
                        $enabledModules[$widget->dir] = $widget;
                    }
                    else {
                        $disabledModules[$widget->dir] = $widget;
                    }
                } // isMozzle
            } // foreach
        }

        return array('enabled' => $enabledModules, 'disabled' => $disabledModules);

    } // getModuleInfo()

    /**
     * Run an action in all enabled modules, taking into account any special
     * rules that are relevant for group-enabled modules
     *
     * @param $controller string The controller name to use (e.g. "index")
     * @param $action string The action name to call (e.g. "searchConfiguration")
     * @param $args array optional array of arguments to pass to the action
     */
    public static function dispatchToEnabledModules($controller, $action, $args = null) {
        $enabledWidgets = XG_ModuleHelper::getEnabledModules();
        /* If groups is enabled, make sure to talk to widget instances that are
         * group-aware but could be inactive on the top level */
        if ($enabledWidgets['groups']) {
            foreach (XG_GroupHelper::groupEnabledWidgetInstanceNames() as $groupEnabledWidgetInstanceName) {
                $enabledWidgets[$groupEnabledWidgetInstanceName] = W_Cache::getWidget($groupEnabledWidgetInstanceName);
            }
        }
        foreach ($enabledWidgets as $widgetName => $widget) {
            if ($widget->controllerHasAction($controller,$action)) {
                list($r,$html) = $widget->dispatch($controller,$action,$args);
            }
        }
    }

    public static $DIVIDER = ' &bull; ';

    /**
     * Forms a string of HTML links to actions based on the currently enabled modules.
     *
     * Return value of the form:
     *
     *    '<a href="blah">Add Photos</a> &bull; <a href="blah2">Upload Videos</a>'
     *
     * where each link refers to an action that can be taken on this network (that is,
     * you won't get a link to add photos if photos are not enabled).
     *
     * @param 	$pageType	string	Must be 'profile' or 'main'.  Determines what links
     *								are included for certain modules.
     * @return 				string 	HTML suitable for inclusion in a template.
     */
    public static function moduleSpecificPhrases($pageType) {
        $s = '';
        $enabled = XG_ModuleHelper::getEnabledModules();
        if ($enabled['photo']) {
            $s .= self::phraseLink('ADD_PHOTOS_SENTENCE_CASE', 'photo', 'photo', XG_MediaUploaderHelper::action());
        }
        if ($enabled['video']) {
            $s .= self::phraseLink('UPLOAD_VIDEOS', 'video', 'video', XG_MediaUploaderHelper::action());
        }
        if ($enabled['forum'] && $pageType === 'profile') {
            $s .= self::phraseLink('CONTRIBUTE_TO_DISCUSSIONS', 'forum', 'index', 'index');
        }
        if ($enabled['forum'] && $pageType === 'main') {
            $s .= self::phraseLink('MANAGE_FORUM_CATEGORIES', 'forum', 'category', 'edit');
            $s .= self::phraseLink('START_DISCUSSIONS', 'forum', 'index', 'index');
        }
        if ($enabled['groups'] && $pageType === 'profile') {
            $s .= self::phraseLink('JOIN_GROUPS', 'groups', 'index', 'index');
        }
        if ($enabled['groups'] && $pageType === 'main') {
            $s .= self::phraseLink('CREATE_GROUPS', 'groups', 'group', 'new');
        }
        return ($s ? mb_substr($s, 0, mb_strlen($s) - mb_strlen(self::$DIVIDER)) : '');
    }

    // Takes a phrase key and makes it an HTML link to the specified action of the
    // specified controller.
    private static function phraseLink($phrase, $module, $controller, $action) {
        $link = W_Cache::getWidget($module)->buildUrl($controller, $action);
        return '<a href="' . xnhtmlentities($link) . '">' . xg_html($phrase) . '</a>' . self::$DIVIDER;
    }
}
