<?php

XG_App::includeFileOnce('/lib/XG_Embed.php');
XG_App::includeFileOnce('/lib/XG_LangHelper.php');

/**
 * Useful functions for HTML output for layouts
 *
 * @see XG_Layout
 */
class XG_LayoutHelper {

    /**
     * Outputs the HTML for the specified layout.
     *
     * @param $xgLayout		XG_Layout	The layout to render.
     * @param $controller W_Controller  The current controller, specifying layoutName and layoutType
     */
    public static function renderLayout($xgLayout, $controller) {
        // TODO: Pass $xgLayout around instead of $controller
		$xgLayout = XG_LayoutHelper::updateSidebarIfNecessary($xgLayout);
		$layout = $xgLayout->getLayout()->documentElement;
        if ($layout) {
	        foreach ($layout->childNodes as $colgroup) {
                if ($colgroup instanceof DOMElement) {
                    self::renderElement($colgroup, $controller, $xgLayout);
                }
            }
        }
    }

	/**
	 * Outputs the HTML for the module is the sitewide sidebar.
	 *
	 * What is in the sidebar is determined by reading the PageLayout object's XML.
	 *
	 * @param	W_Controller	Current controller.
	 * @return 					void
	 */
	public static function renderSitewideSidebarEmbeds(W_Controller $controller) {
		if (! XG_App::appIsLaunched()) { return; }
        $controller->app = XN_Application::load();

        $controller->enabledModules = XG_ModuleHelper::getEnabledModules();
        $controller->disabledModules = XG_ModuleHelper::getDisabledModules();

		$xgLayout = XG_Layout::load('index');
		$layout = $xgLayout->getLayout();
		
		$xpath = new DOMXPath($layout);
		
        //  Use the contents of the 'layout' element within the document
		$controller->layout = $xgLayout->getLayout()->documentElement;
        $controller->layoutName = $xgLayout->getName();
        $controller->layoutType = $xgLayout->getType();

        $nodeList = $xpath->query('/layout/colgroup/column[2]');
		foreach ($nodeList->item(0)->childNodes as $element) {
			if ($element->getAttribute('sitewide')) {
                self::renderElement($element, $controller, $xgLayout);
			}
		}
	}
	
	/**
	 * Read the supplied layout and check if it contains an old style sidebar of the form:
	 *
	 * <module widgetName="main" action="sidebar" embedInstanceId="1"/>
	 *
	 * If so, replace it with module elements for You, Ads, Created By and Badges.
	 *
	 * This function exists to ensure the backwards compatibility of old format PageLayout XML
	 * now that we store sidebar modules (both permanent and moveable) in the XML as well as
	 * main page items instead of just a single "sidebar" module.
	 *
	 * In the past we did not store the sidebar for the whole site in the index PageLayout object in 
	 * the content store.  Just the items to appear on the main page.  Sidebar items were
	 * hardcoded.  To preserve the layouts of those networks that were created before v1.11
	 * we check here for the presence of the old sidebar module.  If it is there we  update the XML
	 * for the sidebar in the PageLayout object in the content store with the items that were 
	 * previously hardcoded.  This change only affects the homepage PageLayout ('index').  
	 * Other page layouts for My Page or Groups still call action_sidebar which (eventually)
	 * renders the main page's sitewide sidebar modules. 
	 * 
	 * This function is also piggybacked to add the Activity dashboard widget to the top of the
	 * center column on every homepage because it is being released at the same time.  If we
	 * need to do this kind of thing again we will have to introduce versioning similar to that
	 * on user pages: 
	 *
	 * @param	$xgLayout	XG_Layout	XG_Layout that contains the site-wide sidebar.
	 * @return  XG_Layout				The supplied layout if no changes are required.  The newly modified layout if changes have been made.
	 */
    public static function updateSidebarIfNecessary($xgLayout) {
		if ($xgLayout->getName() !== 'index') {
			return $xgLayout;
		}
		$xpath = new DOMXPath($xgLayout->getLayout());
		$sidebarPath = '/layout/colgroup/column[2]';
		//TODO this will fail if there is nothing at all in the sidebar, although this is not currently possible.
		$element = $xpath->query($sidebarPath)->item(0)->firstChild;
		if (! $element || $element->getAttribute('action') !== 'sidebar') {
			return $xgLayout;
		}
		$sidebarModulePath = $sidebarPath . '/module';
		$added = 0;
		if ($xgLayout->insertModule('profiles', 'embed1badge', $sidebarModulePath, array('sitewide' => true))) {
			$added += 1;			
		} else { error_log("could not add badges"); }
		if ($xgLayout->insertModule('main', 'embed1createdBy', $sidebarModulePath, array('sitewide' => true))) {
			$added += 1;			
		} else { error_log("could not add createdby"); }
		if ($xgLayout->insertModule('main', 'embed1ads', $sidebarModulePath, array('sitewide' => true))) {
			$added += 1;			
		}
		if ($xgLayout->insertModule('main', 'embed1you', $sidebarModulePath, array('sitewide' => true))) {
			$added += 1;
		} else { error_log("coud not add you"); }
		if ($xgLayout->insertModule('activity', 'embed2', '/layout/colgroup/column/colgroup/column[2]/module', NULL)) {
			$added += 1;
		} else { error_log("could not add activity dashboard"); }
		if ($added === 5) {
			// remove sidebar module from XML
			$xgLayout->removeElement($sidebarModulePath . "[@action='sidebar']");
			$xgLayout = null; // Force save by calling XG_Layout's destructor. TODO Should actually add a save method to xg_layout to make this explicit.
		} else {
			error_log("We tried to update the sidebar from old style to new style but we failed to add all the "
				. "modules that we wanted to add.");
		}
		return XG_Layout::load('index');
	}
	
	/**
	 * Scan the sidebar of the specified layout.  If it does not contain ads/created by and the current network is
	 * not paying for "run your own ads" or "protect your network" then insert ads/created by in the sidebar.
	 *
	 * Devised to prevent users remaining ad-free after they stop paying their "run your own ads" payment or 
	 * similar with "protect your network".
	 *
	 * @param	$xgLayout	XG_Layout	XG_Layout that contains the site-wide sidebar.
	 * @return  XG_Layout				The supplied layout if no changes are required.  The newly modified layout if changes have been made.
	 */
	public static function putPayServicesInSidebarIfNecessary($originalXgLayout) {
		$xgLayout = $originalXgLayout;
		if ($xgLayout->getName() !== 'index') {
			return $xgLayout;
		}
		$runOwnAds = XG_App::runOwnAds();
		$protectYourNetwork = XG_App::protectYourNetwork();
		if (! $protectYourNetwork) {
			$xgLayout = XG_LayoutHelper::putCreatedByInSidebarIfNecessary($xgLayout);
		}
		if (! $runOwnAds) {
			$xgLayout = XG_LayoutHelper::putAdsInSidebarIfNecessary($xgLayout);
		}
		return $xgLayout;
	}
	
	/**
	 * Scan the sidebar for an ads module.  If not found, add it to layout and reload.
	 *
	 * @param	XG_Layout	Layout to scan sidebar of.
	 */
    private static function putAdsInSidebarIfNecessary($xgLayout) {
		if ($xgLayout->getName() !== 'index') {
			return $xgLayout;
		}
		return XG_LayoutHelper::putModuleInSidebarIfNecessary($xgLayout, 'embed1ads');
	}
	
	/**
	 * Scan the sidebar for a "created by" module.  If not found, add it to layout and reload.
	 *
	 * @param	XG_Layout	Layout to scan sidebar of.
	 */
    private static function putCreatedByInSidebarIfNecessary($xgLayout) {
		if ($xgLayout->getName() !== 'index') {
			return $xgLayout;
		}
		return 	XG_LayoutHelper::putModuleInSidebarIfNecessary($xgLayout, 'embed1createdBy');		
	}
	
	/**
	 * Scan the sidebar for a module with the specified action.  If not found, add it to layout and reload.
	 *
	 * @param	XG_Layout	Layout to scan sidebar of.
	 * @param	string		Action to scan for.
	 */
    private static function putModuleInSidebarIfNecessary($xgLayout, $action) {
		$xpath = new DOMXPath($xgLayout->getLayout());
		$sidebarPath = '/layout/colgroup/column[2]';
		$nodeList = $xpath->query($sidebarPath);
		foreach ($nodeList->item(0)->childNodes as $element) {
			if ($element->getAttribute('action') === $action && $element->getAttribute('sitewide') === true) {
				return $xgLayout;
			}
		}
		// Remove any instances of the module that are elsewhere in the layout.
        $modules = $xpath->query("//module[@action='{$action}']");
		for ($i = 0; $i < $modules->length; $i++) {
			$modules->item($i)->parentNode->removeChild($modules->item($i));
		}
		// Put module in below YOU on sidebar
		//TODO this can result in Created By apppearing /above/ ads in rare circumstances.
		if (! $xgLayout->insertModule('main', $action, $sidebarPath . '/module', array('sitewide' => true), TRUE /* insert 2nd */)) {
			error_log("Failed to return $action to it's rightful place in the sidebar.");
		}
		//TODO what it if is on the main page but NOT in the sidebar - it will now appear twice.
		// Would be nice to remove it from the main page.
		return XG_Layout::load('index');
	}
	
    /**
     * Outputs the HTML for the specified module
     *
     * @param $element DOMElement  the module element
     * @param $controller W_Controller  the current controller, specifying layoutName and layoutType
     * @param $layoutName string The name that, with the type,
     *      uniquely identifies a layout instance, e.g., 'index', or the content ID of a User object
     * @param $layoutType string  optional layout type. 'homepage' is the default and
     *      is the front page layout. 'profiles' is the layout for a user profile page.
     * @param $xgLayout XG_Layout  The layout being rendered.
     */
    private static function renderModule(DOMElement $element, $controller, $layoutName, $layoutType, $xgLayout) {
        $widgetName = $element->getAttribute('widgetName');
        if ($_GET['test_embed']) { $widgetName = $_GET['test_embed']; }
        $action = $element->getAttribute('action');
        $isActive = $element->getAttribute('isActive');
		/* Do not show "normal" embeds in the sidebar when not signed in to a private network BAZ-4433 [2007-09-14 bakert] */
		$alwaysShow = array('embed1you', 'embed1ads');
		$hidingPage = (XG_LangHelper::endsWith($_SERVER['SCRIPT_URI'], '/notAllowed') || XG_LangHelper::endsWith($_SERVER['SCRIPT_URI'], '/banned'));
		if ($hidingPage && ! in_array($action, $alwaysShow)) { return; }
        $actionArgs = array('maxEmbedWidth' => $xgLayout->getMaxEmbedWidth($element->parentNode), 'embed' => new XG_Embed($element->getAttribute('embedInstanceId'), $layoutType, $layoutName));
        // Three cases: enabled module, disabled module, permanent module [Jon Aquino 2006-11-18]
        // some group modules now have an isActive attribute, so ensure that it's not set to 0
        if (array_key_exists($widgetName, $controller->disabledModules) && ! $element->getAttribute('forceDisplay')) {
            return;
        }
        try {
            $widget = W_Cache::getWidget($widgetName);
            if ($widget && $widget->controllerHasAction('embed', $action) && $isActive != '0') {
                // Putting $actionArgs inside an array here provides a single argument of "named parameter"
                // array elements to the embed action
                $widget->dispatch('embed', $action, array($actionArgs));
            }
        } catch (Exception $e) {
            // Silently fail and don't render anything if the widget doesn't exist
            // or the widget doesn't have an embed controller
        }
    }

    /** Name of the class containing the renderModule function; overridden by the unit tests. */
    protected static $renderModuleClass = 'XG_LayoutHelper';

    /**
     * Outputs the HTML for the specified element
     *
     * @param $element DOMElement  the element (of type "module", "column", or "colgroup")
     * @param $controller W_Controller  the current controller, specifying layoutName and layoutType
     * @param $xgLayout XG_Layout  The layout being rendered.
     */
    private static function renderElement($element, $controller, $xgLayout) {
        switch ($element->tagName) {
            case 'module':
                call_user_func(array(self::$renderModuleClass, 'renderModule'), $element, $controller, $controller->layoutName, $controller->layoutType, $xgLayout);
                break;
            case 'column':
                self::renderColumn($element, $controller, $xgLayout);
                break;
            case 'colgroup':
                self::renderContainer($element, $controller, $xgLayout);
                break;
        }
    }

    /**
     * Outputs the HTML for the specified container
     *
     * @param $element DOMElement  the container element (e.g., of type "colgroup")
     * @param $controller W_Controller  the current controller, specifying layoutName and layoutType
     * @param $xgLayout XG_Layout  The layout being rendered.
     */
    private static function renderContainer(DOMElement $element, $controller, $xgLayout) {
        echo '<div ';
        $classes = array('xg_' . $element->tagName);
        if (!$element->previousSibling) {
            $classes[] = 'first-child';
        }
        if (!$element->nextSibling) {
            $classes[] = 'last-child';
        }
        echo " class='" . join(' ', $classes) . "'>\n";
        $children = $element->childNodes;
        foreach ($children as $child) {
            self::renderElement($child, $controller, $xgLayout);
        }
        echo "		<!-- div class='clear'></div -->\n";
        echo '</div><!--/xg_' . $element->tagName . "-->\n";   
    }

    /**
     * Outputs the HTML for the specified column
     *
     * @param $element DOMElement  the column element
     * @param $controller W_Controller  the current controller, specifying layoutName and layoutType
     * @param $xgLayout XG_Layout  The layout being rendered.
     */
    private static function renderColumn(DOMElement $element, $controller, $xgLayout) {
        // TODO: Eliminate group-specific logic. Actually, the groups homepage
        // shouldn't use XG_Layout. [Jon Aquino 2008-02-29]
        if ($controller->layoutType == 'groups') {
            $colWidth = $element->getAttribute('width');
            $spanWidth = $colWidth * 4;
            echo '<div ';
            $classes = array('xg_span-' . $spanWidth, 'xg_column');
            // TODO: currently the first inner column should get at 'first-child', but doesn't
            // because the previousSibling test finds the preceeding module on group detail pages
            if (!$element->previousSibling) {
                $classes[] = 'first-child';
            }
            if (!$element->nextSibling && $element->previousSibling) {
                $classes[] = 'last-child';
            }
            echo " class='" . join(' ', $classes) . "'>\n";
            $children = $element->childNodes;
            foreach ($children as $child) {
                self::renderElement($child, $controller, $xgLayout);
            }
            echo "&nbsp;";  // placeholder to enforce width
            echo '</div><!--/xg_' . $element->tagName . "-->\n";
        } else {
            $colWidth = $element->getAttribute('width');
            echo '<div ';
            $classes = array('xg_' . $colWidth . 'col');
            if (!$element->previousSibling) {
                $classes[] = 'first-child';
            }
            if (!$element->nextSibling) {
                $classes[] = 'last-child';
            }
            echo " class='" . join(' ', $classes) . "'>\n";
            $children = $element->childNodes;
            foreach ($children as $child) {
                self::renderElement($child, $controller, $xgLayout);
            }
            echo "&nbsp;";  // placeholder to enforce width
            echo '</div><!--/xg_' . $element->tagName . "-->\n";   
        }
    }

    /**
     * Finds the attributes of module elements
     *
     * @param $element DOMElement  the root node to search
     * @param $attributes  array  array to which to add  the names and values
     *     of the attributes of $element (if it is of type "module") and its child module elements
     */
    public static function getAttributes($element, &$attributes) {
        if ($element->tagName == 'module') {
            $embed = array();
            $attrList = $element->attributes;
            if ($attrList) {
                $i = 0;
                while ($item = $attrList->item($i)) {
                    $embed[$item->name] = $item->value;
                    $i++;
                }
            }
            $attributes[$element->getAttribute('embedInstanceId')] = $embed;
        }
        else {
            foreach ($element->childNodes as $child) {
                self::getAttributes($child, $attributes);
            }
        }
    }

}
