<?php

/**
 * Useful functions for feeds.
 */
class XG_FeedHelper {
    
    const FEED_CACHE_LABEL = 'feed';
    protected static $key;

    /**
     * Outputs the XML for an Atom feed. Assumes that XG_Cache has been
     * primed with the profiles of the content-object contributors.
     *
     * @param $objects array  The content objects to convert into feed items
     * @param $title string  Title of the feed; the app name is automatically added.
     * @param $feedAuthor XN_Profile  The sole contributor to this feed, or NULL if more than one person contributes.
     * @param $descriptionBuilder string  Code to create the XML-encoded description; defaults to (note the trailing semicolon): return xg_xmlentities($object->description);
     * @param $titleBuilder string  Code to create the XML-encoded title
     * @param $publicationDateAttribute string  System or developer attribute for the date on which the object was published; defaults to 'createdDate'.
     */
    public static function outputFeed($objects, $title, $feedAuthor = NULL, $descriptionBuilder = NULL, $titleBuilder = NULL, $publicationDateAttribute = NULL) {
        $descriptionBuilderFunction = create_function('$object', $descriptionBuilder ? $descriptionBuilder : 'return xg_xmlentities($object->description);');
        $titleBuilderFunction = create_function('$object', $titleBuilder ? $titleBuilder : 'return xg_xmlentities($object->type == \'User\' ? xg_username(XG_Cache::profiles($object->title)) : ($object->title ? $object->title : xg_excerpt($object->description, 30)));');
        $feedUrl = XG_HttpHelper::removeParameters(XG_HttpHelper::currentUrl(), array('page'));
        /* BAZ-4877: If there's no feed author, pre-load all the profiles for all the authors of
         * the individual feed entries */
         if (! $feedAuthor) {
             $contributors = array();
             foreach ($objects as $object) {
                 $contributors[$object->contributorName] = $object->contributorName;
             }
             $profiles = XG_Cache::profiles($contributors);
         }
         /* If the feed should be cached, capture output */
         if (mb_strlen(self::$key)) {
             ob_start();
         }
        ?><?xml version="1.0" encoding="utf-8"?>
        <feed xmlns="http://www.w3.org/2005/Atom">
            <title><%= xg_xmlentities($title ? $title . ' - ' : '') %><%= XN_Application::load()->name %></title>
            <link rel="self" href="<%= xg_xmlentities($feedUrl) %>"/>
            <updated><%= xg_xmlentities(gmdate('Y-m-d\TH:i:s\Z')) %></updated>
            <?php
            if ($feedAuthor) { ?>
                <author><name><%= xg_xmlentities(xg_username($feedAuthor)) %></name></author>
            <?php
            } ?>
            <id><%= xg_xmlentities($feedUrl) %></id>
            <?php
            foreach ($objects as $object) { ?>
                <entry>
                    <title><%= $titleBuilderFunction($object) %></title>
                    <link rel="alternate" href="<%= xg_xmlentities('http://' . $_SERVER['HTTP_HOST'] . ($object->type == 'User' ? '/profile/' . User::profileAddress($object->contributorName) : '/xn/detail/' . $object->id )) %>"/>
                    <?php
                    /* From http://diveintomark.org/archives/2004/05/28/howto-atom-id#tag */
                    $atomId = 'tag:' . $_SERVER['HTTP_HOST'] . ',' . mb_substr($publicationDateAttribute ? $object->my->$publicationDateAttribute : $object->createdDate,0,10) . ':' . $object->id; ?>
                    <id><%= $atomId %></id>
                    <?php /* Use createdDate rather than updatedDate, which changes constantly, pushing old items to the top of the feed [Jon Aquino 2007-04-05] */ ?>
                    <updated><%= xg_xmlentities($object->createdDate) %></updated>
                    <?php
                    if (! $feedAuthor) { ?>
                        <author><name><%= xg_xmlentities(xg_username($profiles[$object->contributorName])) %></name></author>
                    <?php
                    } ?>
                    <summary type="html">
                        <%= $descriptionBuilderFunction($object) %>
                    </summary>
                </entry>
            <?php
            } ?>
        </feed>
    <?php
    
        /* If the feed should be cached, put it in the cache and then output it */
        if (mb_strlen(self::$key)) {
            $feedXml = ob_get_clean();
            try {
                XN_Cache::put(self::$key, $feedXml, self::FEED_CACHE_LABEL);
            } catch (Exception $e) {
            }
            echo $feedXml;
        }
    }
    
    /**
     * Cache the feed generated by a subsequent call to outputFeed() for
     * $maxAge seconds
     *
     * @param $key string cache key
     * @param $maxAge integer how many seconds to cache the feed for
     */
    public static function cacheFeed($vars, $getVars = null, $maxAge = 1800) {
        self::$key = self::cacheKey($vars, $getVars);
        try {
            $feed = XN_Cache::get(self::$key, $maxAge);
            if (mb_strlen($feed)) {
                echo $feed;
                exit();
            }
            if ($_GET['test_caching']) { var_dump('Not cached'); }
        } catch (Exception $e) {
            /* If something went wrong with the cache request, then
             * proceed if it's a cache miss */
        }
    }
    
    protected static function cacheKey($vars, $getVars = null) {
        $key = 'feed-';
        foreach ($vars as $k => $v) {
            $key .= "$k-$v,";
        }
        if (is_array($getVars)) {
            foreach ($getVars as $k) {
                $v = isset($_GET[$k]) ? $_GET[$k] : '';
                $key .= "$k-$v,";
            }
        }
        return $key;
    }
}
